#!/usr/bin/env python3
"""
Simple workflow runner: generates step-by-step prompt packs for a given YAML workflow.

Features:
- Loads a workflow YAML (workflows/*.yaml)
- Optionally loads models/model_suggestions.yaml
- Interactively collects required inputs (or reads from a JSON file)
- Renders per-step prompt files for lead/reviewer actors

Dependencies: PyYAML (pip install pyyaml)
Usage:
  python scripts/run_workflow.py --workflow workflows/texterstellung.yaml \
         --models models/model_suggestions.yaml --out runs

  Optional: --inputs inputs.json (pre-filled answers to the workflow "inputs" fields)
"""

import argparse
import datetime as dt
import json
import os
import re
import sys
from pathlib import Path

try:
    import yaml  # type: ignore
except Exception:
    print("PyYAML is required. Install with: pip install pyyaml", file=sys.stderr)
    sys.exit(1)


def slugify(text: str) -> str:
    s = re.sub(r"[^a-zA-Z0-9]+", "-", text.strip().lower()).strip("-")
    return s or "workflow"


def load_yaml(path: Path):
    with path.open("r", encoding="utf-8") as f:
        return yaml.safe_load(f)


def ensure_dir(p: Path):
    p.mkdir(parents=True, exist_ok=True)


def prompt_inputs(workflow_inputs: dict, preset: dict | None) -> dict:
    answers = {}
    for key, label in workflow_inputs.items():
        if preset and key in preset:
            answers[key] = preset[key]
            continue
        print(safe_console_text(f"Provide input for '{key}': {label}"))
        print(safe_console_text("Enter text. Finish with an empty line."))
        lines = []
        while True:
            try:
                line = input()
            except EOFError:
                line = ""
            if line.strip() == "":
                break
            lines.append(line)
        answers[key] = "\n".join(lines).strip()
    return answers


def _resolve_wf_suggestions(suggestions: dict | None, wf_slug: str) -> dict | None:
    if not suggestions:
        return None
    pool = suggestions.get("workflows", {}) if isinstance(suggestions, dict) else {}
    if not isinstance(pool, dict):
        return None
    # Try several keys: exact slug, slug with _/-, and plain
    candidates = [
        wf_slug,
        wf_slug.replace('-', '_'),
        wf_slug.replace('_', '-'),
    ]
    for c in candidates:
        node = pool.get(c)
        if node:
            return node
    # As a last resort, try keys that loosely match ignoring _/-
    key_norm = wf_slug.replace('-', '').replace('_', '')
    for k, v in pool.items():
        if k.replace('-', '').replace('_', '') == key_norm:
            return v
    return None


def select_models(suggestions: dict | None, wf_slug: str, overrides: dict) -> dict:
    selected = {}
    wf_sugs = _resolve_wf_suggestions(suggestions, wf_slug)
    # map known roles to suggestion lists
    role_keys = [
        "lead_ai",
        "reviewer_ai",
        "prompt_engineer_ai",
        "image_generator",
    ]
    if wf_sugs:
        for rk in role_keys:
            if rk in wf_sugs:
                lst = wf_sugs[rk]
                # pick preferred if flagged, else first
                preferred = next((x for x in lst if x.get("preferred")), None)
                choice = preferred or (lst[0] if lst else None)
                selected[rk] = choice
    # apply overrides
    for k, v in overrides.items():
        if v:
            selected[k] = {"provider": v, "notes": "user override"}
    return selected


def render_prompt(step: dict, inputs: dict, actor_label: str, model: str | None) -> str:
    header = [
        f"Actor: {actor_label}",
        f"Model: {model or 'TBD'}",
        f"Step: {step.get('id', 'unknown')}",
        "",
    ]
    body = []
    if "do" in step and step["do"]:
        body.append("Task:")
        body.append(step["do"].strip())
        body.append("")
    if "checklist" in step and step["checklist"]:
        body.append("Checklist (verify before handoff):")
        for item in step["checklist"]:
            body.append(f"- {item}")
        body.append("")
    if "rubric" in step and step["rubric"]:
        body.append("Reviewer rubric (score and critique):")
        for item in step["rubric"]:
            body.append(f"- {item}")
        body.append("")
    # Inputs context
    body.append("Context (inputs):")
    for k, v in inputs.items():
        body.append(f"- {k}: {v}")
    body.append("")
    body.append("Deliverable:")
    if "rubric" in step:
        body.append("- Provide a prioritized issue list with fixes and rationale.")
    else:
        body.append("- Provide the artifact(s) and a brief change summary.")
    return "\n".join(header + body)


def run_workflow(wf_path: Path, models_path: Path | None, out_base: Path, preset_inputs: dict | None, overrides: dict) -> Path:
    if not wf_path.exists():
        print(safe_console_text(f"Workflow file not found: {wf_path}"), file=sys.stderr)
        sys.exit(2)

    wf = load_yaml(wf_path)
    wf_name = wf.get("name", wf_path.stem)
    wf_slug = slugify(wf_path.stem)

    suggestions = None
    if models_path:
        if not models_path.exists():
            print(safe_console_text(f"Model suggestions not found: {models_path}"), file=sys.stderr)
            sys.exit(2)
        suggestions = load_yaml(models_path)

    # Inputs
    inputs_schema = wf.get("inputs", {})
    if not isinstance(inputs_schema, dict) or not inputs_schema:
        print(safe_console_text("Workflow has no inputs schema."), file=sys.stderr)
        inputs = {}
    else:
        inputs = prompt_inputs(inputs_schema, preset_inputs)

    selected = select_models(suggestions, wf_slug, overrides)

    # Prepare run dir
    ts = dt.datetime.now().strftime("%Y%m%d-%H%M%S")
    run_dir = out_base / f"{ts}-{wf_slug}"
    ensure_dir(run_dir)

    # Save metadata
    meta = {
        "workflow": {"path": str(wf_path), "name": wf_name, "slug": wf_slug},
        "models": selected,
        "inputs": inputs,
    }
    (run_dir / "meta.json").write_text(json.dumps(meta, indent=2, ensure_ascii=False), encoding="utf-8")

    # Write README
    readme = [
        f"Workflow: {wf_name}",
        "",
        "How to use these prompts:",
        "- For each numbered file, copy the prompt into your chosen AI chat.",
        "- Use the indicated model for the actor (lead/reviewer/prompt engineer).",
        "- Paste the AI's output back into this run folder as step_output.md (optional).",
        "",
        "Selected models:",
    ]
    for role, info in selected.items():
        if not info:
            continue
        readme.append(f"- {role}: {info.get('provider')} ({info.get('notes','')})")
    (run_dir / "README.txt").write_text("\n".join(readme), encoding="utf-8")

    # Generate prompts for steps
    steps = wf.get("steps", [])
    if not steps:
        print(safe_console_text("Workflow defines no steps."), file=sys.stderr)
        sys.exit(3)

    def role_for_step(step: dict) -> str:
        return step.get("actor", "lead_ai")

    for idx, step in enumerate(steps, start=1):
        role = role_for_step(step)
        model_name = None
        if role in selected and selected[role]:
            model_name = selected[role].get("provider")
        actor_label = {
            "lead_ai": "Lead AI",
            "reviewer_ai": "Reviewer AI",
            "prompt_engineer_ai": "Prompt Engineer AI",
            "image_generator": "Image Generator",
        }.get(role, role)

        fname = f"{idx:02d}_{step.get('id','step')}_{slugify(actor_label)}.txt"
        prompt_text = render_prompt(step, inputs, actor_label, model_name)
        (run_dir / fname).write_text(prompt_text, encoding="utf-8")

    print(safe_console_text(f"Prompt pack created: {run_dir}"))
    return run_dir


def interactive_wizard(default_models_path: Path | None, out_base: Path):
    print(safe_console_text("=== KI-Matrix Workflow Wizard ==="))
    # Discover workflows
    wf_dir = Path("workflows")
    available = {
        "1": ("Texterstellung", wf_dir / "texterstellung.yaml"),
        "2": ("Bildgenerierung", wf_dir / "bildgenerierung.yaml"),
        "3": ("Programmierung Web", wf_dir / "web_programming.yaml"),
        "4": ("HTML & CSS", wf_dir / "html_css.yaml"),
        "5": ("TYPO3", wf_dir / "typo3.yaml"),
        "6": ("WordPress", wf_dir / "wordpress.yaml"),
    }
    for key, (label, path) in available.items():
        status = "OK" if path.exists() else "MISSING"
        print(safe_console_text(f"{key}. {label} [{path}] - {status}"))
    print(safe_console_text("0. Abbrechen"))

    choice = input("Auswahl (0-6): ").strip()
    if choice == "0" or choice not in available:
        print(safe_console_text("Abgebrochen."))
        sys.exit(0)
    label, wf_path = available[choice]

    # Model suggestions
    if default_models_path and default_models_path.exists():
        models_path = default_models_path
    else:
        # Ask for an optional path or leave empty
        mp = input("Pfad zu model_suggestions.yaml (Enter für überspringen): ").strip()
        models_path = Path(mp) if mp else None

    # Optional inputs.json
    ip = input("Pfad zu inputs.json (Enter für manuelle Eingabe): ").strip()
    preset = None
    if ip:
        try:
            with open(ip, "r", encoding="utf-8") as f:
                preset = json.load(f)
        except Exception as e:
            print(safe_console_text(f"Konnte inputs.json nicht laden: {e}"))
            preset = None

    # Optional overrides via numeric selection from suggestions
    overrides: dict = {}
    wf_slug = slugify(wf_path.stem)
    suggestions = None
    if models_path and models_path.exists():
        try:
            suggestions = load_yaml(models_path)
        except Exception as e:
            print(safe_console_text(f"Konnte Modellsuggestions nicht laden: {e}"))
            suggestions = None

    wf_sugs = _resolve_wf_suggestions(suggestions, wf_slug)
    role_labels = {
        "lead_ai": "Lead AI",
        "reviewer_ai": "Reviewer AI",
        "prompt_engineer_ai": "Prompt Engineer AI",
        "image_generator": "Image Generator",
    }

    def pick_role(role_key: str):
        if not wf_sugs or role_key not in wf_sugs:
            # free text fallback
            val = input(f"{role_labels.get(role_key, role_key)} Provider (Enter = automatisch): ").strip()
            if val:
                overrides[role_key] = val
            return
        lst = wf_sugs[role_key]
        print(safe_console_text(f"\n{role_labels.get(role_key, role_key)} Auswahl:"))
        preferred_idx = None
        for idx, item in enumerate(lst, start=1):
            tag = " (preferred)" if item.get("preferred") else ""
            if item.get("preferred"):
                preferred_idx = idx
            notes = f" – {item.get('notes','')}" if item.get('notes') else ""
            print(safe_console_text(f"  {idx}. {item.get('provider','?')}{tag}{notes}"))
        print(safe_console_text("Enter = bevorzugtes Modell wählen (falls markiert), sonst Nr. 1"))
        ans = input(f"Nummer 1-{len(lst)} oder eigener Provider: ").strip()
        if not ans:
            # default to preferred or first
            if preferred_idx is not None:
                choice = lst[preferred_idx-1]
            else:
                choice = lst[0]
            overrides[role_key] = choice.get("provider")
            return
        if ans.isdigit():
            i = int(ans)
            if 1 <= i <= len(lst):
                overrides[role_key] = lst[i-1].get("provider")
                return
        # Fallback: treat as custom provider string
        overrides[role_key] = ans

    print(safe_console_text("\nModelle auswählen (leer = bevorzugtes Modell):"))
    # Always offer lead/reviewer; the other two are relevant for Bildgenerierung
    pick_role("lead_ai")
    pick_role("reviewer_ai")
    pick_role("prompt_engineer_ai")
    pick_role("image_generator")

    # Run
    run_workflow(wf_path, models_path, out_base, preset, overrides)


def main():
    ap = argparse.ArgumentParser(description="Generate prompt pack from a workflow YAML or run interactive wizard.")
    ap.add_argument("--workflow", help="Path to workflow YAML (omit to open wizard)")
    ap.add_argument("--models", help="Path to model_suggestions.yaml")
    ap.add_argument("--inputs", help="Path to JSON file with answers for workflow inputs")
    ap.add_argument("--out", default="runs", help="Output base directory")
    ap.add_argument("--lead-ai", dest="lead_ai", help="Override lead AI provider name")
    ap.add_argument("--reviewer-ai", dest="reviewer_ai", help="Override reviewer AI provider name")
    ap.add_argument("--prompt-ai", dest="prompt_ai", help="Override prompt engineer AI provider name")
    ap.add_argument("--image-generator", dest="image_generator", help="Override image generator provider")
    ap.add_argument("--interactive", action="store_true", help="Force interactive wizard")
    args = ap.parse_args()

    out_base = Path(args.out)
    ensure_dir(out_base)

    # Interactive wizard if requested or if no workflow provided
    default_models = Path(args.models) if args.models else (Path("models") / "model_suggestions.yaml")
    if args.interactive or not args.workflow:
        interactive_wizard(default_models if default_models.exists() else None, out_base)
        return

    # Non-interactive path (CLI-driven)
    wf_path = Path(args.workflow)
    models_path = Path(args.models) if args.models else None
    preset = None
    if args.inputs:
        with open(args.inputs, "r", encoding="utf-8") as f:
            preset = json.load(f)
    overrides = {
        "lead_ai": args.lead_ai,
        "reviewer_ai": args.reviewer_ai,
        "prompt_engineer_ai": args.prompt_ai,
        "image_generator": args.image_generator,
    }
    run_workflow(wf_path, models_path, out_base, preset, overrides)


def safe_console_text(text: str) -> str:
    """Make text printable on consoles with limited encodings (e.g., cp1252).
    Falls back to backslash escapes for unsupported characters.
    """
    enc = getattr(sys.stdout, "encoding", None) or "utf-8"
    try:
        return text.encode(enc, errors="backslashreplace").decode(enc, errors="ignore")
    except Exception:
        # As a last resort, strip non-ASCII
        return text.encode("ascii", errors="backslashreplace").decode("ascii")

if __name__ == "__main__":
    main()
