// v2: Minimal Node server for Google Wallet Generic Pass (no external deps)
// Provides class/object previews and a Save-to-Wallet JWT endpoint

const http = require('http');
const fs = require('fs');
const path = require('path');
const url = require('url');
const crypto = require('crypto');

const PORT = process.env.PORT || 3100;
const ROOT = __dirname;

// Load config
const CFG_PATH = path.join(ROOT, 'config.json');
let CFG = {
  issuerId: '',
  classId: '',
  hexBackgroundColor: '#4285f4',
  logo: { uri: '', description: '' },
  hero: { uri: '', description: '' },
  serviceAccountKeyPath: './key.json',
  // Default card rows referencing textModulesData ids
  classRows: [
    {
      twoItems: {
        startItem: { firstValue: { fields: [ { fieldPath: "object.textModulesData['telefon']" } ] } },
        endItem:   { firstValue: { fields: [ { fieldPath: "object.textModulesData['e-mail']" } ] } }
      }
    },
    {
      oneItem: {
        item: { firstValue: { fields: [ { fieldPath: "object.textModulesData['mobil']" } ] } }
      }
    },
    {
      twoItems: {
        startItem: { firstValue: { fields: [ { fieldPath: "object.textModulesData['plz']" } ] } },
        endItem:   { firstValue: { fields: [ { fieldPath: "object.textModulesData['ort']" } ] } }
      }
    }
  ]
};
try {
  const raw = fs.readFileSync(CFG_PATH, 'utf8');
  const obj = JSON.parse(raw);
  CFG = Object.assign(CFG, obj);
} catch (e) {
  console.warn('[v2] Warn: config.json missing or invalid, using defaults');
}

// Load service account key
let serviceKey = null;
try {
  const keyPath = path.resolve(ROOT, CFG.serviceAccountKeyPath || './key.json');
  const raw = fs.readFileSync(keyPath, 'utf8');
  serviceKey = JSON.parse(raw);
} catch (e) {
  console.warn('[v2] Warn: service account key not loaded:', e.message);
}

function sendJson(res, status, obj) {
  const body = JSON.stringify(obj);
  res.writeHead(status, {
    'Content-Type': 'application/json; charset=utf-8',
    'Content-Length': Buffer.byteLength(body)
  });
  res.end(body);
}

function sendFile(res, filePath) {
  fs.readFile(filePath, (err, data) => {
    if (err) {
      res.writeHead(404, { 'Content-Type': 'text/plain; charset=utf-8' });
      return res.end('Not Found');
    }
    const ext = path.extname(filePath).toLowerCase();
    const types = { '.html': 'text/html; charset=utf-8', '.css': 'text/css', '.js': 'application/javascript' };
    res.writeHead(200, { 'Content-Type': types[ext] || 'application/octet-stream' });
    res.end(data);
  });
}

function parseBody(req) {
  return new Promise((resolve) => {
    let data = '';
    req.on('data', (chunk) => { data += chunk; });
    req.on('end', () => {
      try {
        const isJson = (req.headers['content-type'] || '').includes('application/json');
        resolve(isJson ? JSON.parse(data || '{}') : { raw: data });
      } catch (_) {
        resolve({});
      }
    });
  });
}

function b64url(input) {
  return Buffer.from(input).toString('base64').replace(/=/g, '').replace(/\+/g, '-').replace(/\//g, '_');
}

function createJwt(payloadObj, privateKey) {
  const header = { alg: 'RS256', typ: 'JWT' };
  const headerB64 = b64url(JSON.stringify(header));
  const payloadB64 = b64url(JSON.stringify(payloadObj));
  const toSign = `${headerB64}.${payloadB64}`;
  const signer = crypto.createSign('RSA-SHA256');
  signer.update(toSign);
  const signature = signer.sign(privateKey);
  const sigB64 = signature.toString('base64').replace(/=/g, '').replace(/\+/g, '-').replace(/\//g, '_');
  return `${toSign}.${sigB64}`;
}

function slugify(s) {
  try {
    return String(s || '')
      .toLowerCase()
      .normalize('NFKD')
      .replace(/[^a-z0-9]+/g, '-')
      .replace(/(^-|-$)/g, '');
  } catch (_) {
    return '';
  }
}

function buildClassJson(overrides = {}) {
  const issuerId = String((overrides.issuerId ?? CFG.issuerId) || '').trim();
  const classId = String((overrides.classId ?? CFG.classId) || '').trim();
  const rows = Array.isArray(overrides.classRows) ? overrides.classRows : (Array.isArray(CFG.classRows) ? CFG.classRows : []);
  return {
    id: `${issuerId}.${classId}`,
    classTemplateInfo: {
      cardTemplateOverride: {
        cardRowTemplateInfos: rows
      }
    }
  };
}

function buildObjectJson(input = {}) {
  const issuerId = String(CFG.issuerId || '').trim();
  const classId = String(CFG.classId || '').trim();
  const id = input.objectId
    ? `${issuerId}.${String(input.objectId).trim()}`
    : `${issuerId}.${Math.floor(Date.now() / 1000)}-${Math.random().toString(36).slice(2, 8)}`;

  // Ensure 3 text modules as per example, allow override and normalize
  const rawTextModules = (Array.isArray(input.textModulesData) && input.textModulesData.length)
    ? input.textModulesData
    : [
        { id: 'telefon', header: 'Telefon', body: '' },
        { id: 'e-mail',  header: 'E-Mail',  body: '' },
        { id: 'mobil',   header: 'Mobil',   body: '' },
        { id: 'plz',     header: 'PLZ',     body: '' },
        { id: 'ort',     header: 'Ort',     body: '' }
      ];
  let autoIdx = 1;
  const textModules = rawTextModules.map(tm => {
    const header = (tm && tm.header != null) ? String(tm.header) : 'HEADER';
    const body = (tm && tm.body != null) ? String(tm.body) : '';
    let id = (tm && tm.id != null) ? String(tm.id) : '';
    if (!id) {
      id = slugify(header) || `field-${autoIdx++}`;
    }
    return { id, header, body };
  });

  const obj = {
    id,
    classId: `${issuerId}.${classId}`,
    logo: {
      sourceUri: { uri: input.logoUri || CFG.logo.uri || '' },
      contentDescription: { defaultValue: { language: 'en-US', value: input.logoDescription || CFG.logo.description || '' } }
    },
    cardTitle: { defaultValue: { language: 'en-US', value: input.cardTitle || 'Dies ist ein Name' } },
    subheader:  { defaultValue: { language: 'en-US', value: input.subheader  || 'Attendee' } },
    header:     { defaultValue: { language: 'en-US', value: input.header     || 'Alex McJacobs' } },
    textModulesData: textModules,
    barcode: {
      type: 'QR_CODE',
      value: String(input.barcodeValue || 'BARCODE_VALUE'),
      alternateText: String(input.barcodeAltText || 'QR-Code')
    },
    hexBackgroundColor: input.hexBackgroundColor || CFG.hexBackgroundColor || '#4285f4',
    heroImage: {
      sourceUri: { uri: input.heroUri || CFG.hero.uri || '' },
      contentDescription: { defaultValue: { language: 'en-US', value: input.heroDescription || CFG.hero.description || '' } }
    }
  };
  // Also emit imageModulesData for better compatibility with GenericObject rendering
  const heroUri = input.heroUri || CFG.hero.uri || '';
  if (heroUri) {
    obj.imageModulesData = [ { mainImage: { sourceUri: { uri: heroUri }, contentDescription: { defaultValue: { language: 'en-US', value: input.heroDescription || CFG.hero.description || '' } } } } ];
  }
  // Optional clickable link below barcode
  if (input.linkUri) {
    obj.linksModuleData = { uris: [ { uri: String(input.linkUri), description: input.linkDescription || 'Link' } ] };
  }
  return obj;
}

async function handleClassPreview(req, res) {
  let body = {};
  try { body = await parseBody(req); } catch (_) { body = {}; }
  const overrides = {};
  if (body && typeof body === 'object') {
    if (body.issuerId) overrides.issuerId = body.issuerId;
    if (body.classId) overrides.classId = body.classId;
    if (Array.isArray(body.classRows)) overrides.classRows = body.classRows;
  }
  return sendJson(res, 200, buildClassJson(overrides));
}

async function handleObjectPreview(req, res) {
  const body = await parseBody(req);
  const obj = buildObjectJson(body || {});
  return sendJson(res, 200, obj);
}

async function handleWalletSave(req, res) {
  if (!serviceKey || !serviceKey.private_key || !serviceKey.client_email) {
    return sendJson(res, 500, { error: 'Service account key missing or invalid. Configure serviceAccountKeyPath.' });
  }
  if (!CFG.issuerId || !CFG.classId) {
    return sendJson(res, 400, { error: 'Missing issuerId or classId in config.json' });
  }
  const body = await parseBody(req);
  const objectJson = buildObjectJson(body || {});
  // Include class definition in JWT so Google applies your card rows
  const classOverrides = {};
  if (body && typeof body === 'object') {
    if (body.issuerId) classOverrides.issuerId = body.issuerId;
    if (body.classId) classOverrides.classId = body.classId;
    if (Array.isArray(body.classRows)) classOverrides.classRows = body.classRows;
  }
  const classJson = buildClassJson(classOverrides);
  const jwtPayload = {
    iss: serviceKey.client_email,
    aud: 'google',
    typ: 'savetoandroidpay',
    iat: Math.floor(Date.now() / 1000),
    payload: { genericClasses: [classJson], genericObjects: [objectJson] }
  };
  try {
    const jwt = createJwt(jwtPayload, serviceKey.private_key);
    const saveUrl = `https://pay.google.com/gp/v/save/${jwt}`;
    return sendJson(res, 200, { saveUrl, objectPreview: objectJson });
  } catch (e) {
    return sendJson(res, 500, { error: e.message || String(e) });
  }
}

function normalizePath(p) {
  if (!p) return '/';
  // strip query already handled by url.parse
  // remove trailing slash (but keep root '/')
  if (p.length > 1 && p.endsWith('/')) p = p.slice(0, -1);
  // help users migrating: allow /v2/* and map to /*
  if (p.startsWith('/v2/')) p = p.slice(3);
  return p || '/';
}

const server = http.createServer(async (req, res) => {
  const { pathname } = url.parse(req.url);
  const route = normalizePath(pathname);
  if (req.method === 'GET' && route === '/') {
    return sendFile(res, path.join(ROOT, 'index.html'));
  }
  if (req.method === 'GET' && route === '/health') {
    return sendJson(res, 200, { ok: true });
  }
  if (req.method === 'GET' && route === '/help') {
    return sendJson(res, 200, {
      ok: true,
      message: 'Wallet v2 standalone running',
      endpoints: [
        'GET /health',
        'GET /config',
        'POST /class/preview',
        'POST /object/preview',
        'POST /wallet/save'
      ],
      port: PORT
    });
  }
  if (req.method === 'GET' && route === '/config') {
    // return a safe subset
    return sendJson(res, 200, {
      issuerId: CFG.issuerId,
      classId: CFG.classId,
      hexBackgroundColor: CFG.hexBackgroundColor,
      logo: CFG.logo,
      hero: CFG.hero
    });
  }
  if (req.method === 'POST' && route === '/class/preview') {
    return handleClassPreview(req, res);
  }
  if (req.method === 'POST' && route === '/object/preview') {
    return handleObjectPreview(req, res);
  }
  if (req.method === 'POST' && route === '/wallet/save') {
    return handleWalletSave(req, res);
  }
  res.writeHead(404, { 'Content-Type': 'application/json; charset=utf-8' });
  res.end(JSON.stringify({
    error: 'Not Found',
    path: route,
    hint: 'Use the documented endpoints on port ' + PORT,
    endpoints: ['GET /health','GET /config','POST /class/preview','POST /object/preview','POST /wallet/save']
  }));
});

server.listen(PORT, () => {
console.log(`Server running at http://localhost:${PORT}`);
});
