﻿const state = {
  brand: {
    company: 'WOW Wallet',
    primaryColor: '#1a73e8',
  },
  person: {
    name: 'Max Mustermann',
    title: 'Customer Success Manager',
    phone: '+49 170 1234567',
    email: 'max@wow-wallet.de',
    photoUrl: '/images/profile-placeholder.svg',
  },
  content: {
    headline: 'Dein digitales WOW Wallet',
    subheadline: 'Speichere alle wichtigen Kontaktdaten direkt in deinem Wallet Pass.',
  },
  links: {
    walletPassUrl: '',
    website: '',
    facebook: '',
    instagram: '',
  },
  qrCode: '',
};

const previewCard = document.getElementById('preview-card');
const companyName = document.getElementById('company-name');
const profilePhoto = document.getElementById('profile-photo');
const headline = document.getElementById('headline');
const subheadline = document.getElementById('subheadline');
const personName = document.getElementById('person-name');
const personTitle = document.getElementById('person-title');
const phoneLink = document.getElementById('phone-link');
const emailLink = document.getElementById('email-link');
const walletButton = document.getElementById('wallet-button');
const qrImage = document.getElementById('qr-code');
const socialLinks = document.getElementById('social-links');
const form = document.getElementById('config-form');

let qrUpdateTimer = null;

function updateBrandColor(color) {
  const fallback = '#1a73e8';
  const value = color && /^#([0-9a-f]{3}){1,2}$/i.test(color) ? color : fallback;
  document.documentElement.style.setProperty('--brand-color', value);
  previewCard.style.setProperty('--brand-color', value);
}

function updateContactLink(element, value, prefix) {
  const container = element.parentElement;
  if (!value) {
    container.style.display = 'none';
    element.removeAttribute('href');
    element.textContent = '';
    return;
  }

  container.style.display = '';
  const normalized = value.replace(/\s+/g, '');
  element.href = `${prefix}${normalized}`;
  element.textContent = value;
}

function renderSocialLinks() {
  const entries = [
    { key: 'website', label: 'Webseite' },
    { key: 'facebook', label: 'Facebook' },
    { key: 'instagram', label: 'Instagram' },
  ];

  socialLinks.innerHTML = '';
  const fragment = document.createDocumentFragment();

  entries.forEach(({ key, label }) => {
    const value = state.links[key];
    if (!value) {
      return;
    }

    const anchor = document.createElement('a');
    anchor.href = value;
    anchor.textContent = label;
    anchor.target = '_blank';
    anchor.rel = 'noopener noreferrer';
    fragment.appendChild(anchor);
  });

  if (!fragment.childNodes.length) {
    socialLinks.style.display = 'none';
  } else {
    socialLinks.style.display = 'flex';
    socialLinks.appendChild(fragment);
  }
}

function applyState() {
  companyName.textContent = state.brand.company || 'Deine Firma';
  headline.textContent = state.content.headline || 'Dein digitales Wallet';
  subheadline.textContent =
    state.content.subheadline || 'Hinterlege Kontaktdaten mit nur einem Tap in deinem Wallet.';

  personName.textContent = state.person.name || '';
  if (state.person.title) {
    personTitle.textContent = state.person.title;
    personTitle.style.display = 'block';
  } else {
    personTitle.textContent = '';
    personTitle.style.display = 'none';
  }

  const fallbackPhoto = '/images/profile-placeholder.svg';
  profilePhoto.src = state.person.photoUrl || fallbackPhoto;
  profilePhoto.alt = state.person.name ? `Profilfoto von ${state.person.name}` : 'Profilfoto';

  updateContactLink(phoneLink, state.person.phone, 'tel:');
  updateContactLink(emailLink, state.person.email, 'mailto:');

  if (state.links.walletPassUrl) {
    walletButton.href = state.links.walletPassUrl;
    walletButton.classList.remove('is-disabled');
  } else {
    walletButton.href = '#';
    walletButton.classList.add('is-disabled');
  }

  if (state.qrCode) {
    qrImage.src = state.qrCode;
    qrImage.alt = 'QR-Code zum Wallet Pass';
    qrImage.style.visibility = 'visible';
  } else {
    qrImage.removeAttribute('src');
    qrImage.style.visibility = 'hidden';
  }

  renderSocialLinks();
  updateBrandColor(state.brand.primaryColor);
}

function updateState(path, value) {
  const segments = path.split('.');
  let cursor = state;
  for (let i = 0; i < segments.length - 1; i += 1) {
    cursor = cursor[segments[i]];
  }
  cursor[segments[segments.length - 1]] = value;
}

async function refreshQrCode(walletUrl) {
  if (!walletUrl) {
    state.qrCode = '';
    applyState();
    return;
  }

  try {
    const response = await fetch(`/api/qr?value=${encodeURIComponent(walletUrl)}`);
    if (!response.ok) {
      throw new Error('QR-Code konnte nicht geladen werden');
    }
    const data = await response.json();
    state.qrCode = data.dataUrl;
    applyState();
  } catch (error) {
    console.error(error);
  }
}

function scheduleQrUpdate(walletUrl) {
  if (qrUpdateTimer) {
    clearTimeout(qrUpdateTimer);
  }

  qrUpdateTimer = setTimeout(() => {
    refreshQrCode(walletUrl);
  }, 300);
}

function handleInput(event) {
  const { dataset, value, type } = event.target;
  const path = dataset.path;
  if (!path) {
    return;
  }

  let nextValue = value;
  if (type === 'url') {
    nextValue = value.trim();
  }

  updateState(path, nextValue);
  applyState();

  if (path === 'links.walletPassUrl') {
    scheduleQrUpdate(nextValue);
  }
}

function populateForm() {
  form.querySelectorAll('[data-path]').forEach((field) => {
    const path = field.dataset.path.split('.');
    let cursor = state;
    path.forEach((segment) => {
      cursor = cursor?.[segment];
    });

    if (typeof cursor === 'string') {
      field.value = cursor;
    }
  });
}

async function bootstrap() {
  try {
    const response = await fetch('/api/profile');
    if (response.ok) {
      const profile = await response.json();
      Object.assign(state.brand, profile.brand || {});
      Object.assign(state.person, profile.person || {});
      Object.assign(state.content, profile.content || {});
      Object.assign(state.links, profile.links || {});
      state.qrCode = profile.qrCode || '';
    }
  } catch (error) {
    console.error('Profil konnte nicht geladen werden', error);
  }

  populateForm();
  form.addEventListener('input', handleInput);
  applyState();
}

document.addEventListener('DOMContentLoaded', bootstrap);
