﻿const path = require('path');
const express = require('express');
const fs = require('fs/promises');
const QRCode = require('qrcode');

const app = express();
const PORT = process.env.PORT || 3000;
const CONFIG_PATH = path.join(__dirname, '..', 'config', 'profile.json');

let cachedProfile = null;
let cachedProfileMtime = null;

async function loadProfile() {
  const stat = await fs.stat(CONFIG_PATH);
  const mtime = stat.mtimeMs;

  if (cachedProfile && cachedProfileMtime === mtime) {
    return cachedProfile;
  }

  const fileContent = await fs.readFile(CONFIG_PATH, 'utf-8');
  const parsed = JSON.parse(fileContent);

  cachedProfile = parsed;
  cachedProfileMtime = mtime;
  return parsed;
}

async function buildResponse(profile) {
  const walletUrl = profile?.links?.walletPassUrl;
  const qrCode = walletUrl ? await QRCode.toDataURL(walletUrl) : null;

  return {
    ...profile,
    qrCode,
  };
}

app.use(express.json());
app.use(express.static(path.join(__dirname, '..', 'public')));

app.get('/api/profile', async (req, res) => {
  try {
    const profile = await loadProfile();
    const response = await buildResponse(profile);
    res.json(response);
  } catch (error) {
    console.error('Failed to load profile configuration', error);
    res.status(500).json({ error: 'Die Profilkonfiguration konnte nicht geladen werden.' });
  }
});

app.get('/api/qr', async (req, res) => {
  const { value } = req.query;

  if (!value) {
    return res.status(400).json({ error: 'Parameter "value" ist erforderlich.' });
  }

  try {
    const dataUrl = await QRCode.toDataURL(value);
    res.json({ dataUrl });
  } catch (error) {
    console.error('Failed to generate QR code', error);
    res.status(500).json({ error: 'QR-Code konnte nicht generiert werden.' });
  }
});

app.get('/health', (_req, res) => {
  res.json({ status: 'ok' });
});

app.listen(PORT, () => {
  console.log(`Wallet landing page server listening on http://localhost:${PORT}`);
});
