# Wallet v2 – Domain-Driven Design (DDD) Entwurf

## Zweck und Scope
- Ziel: Lokales Erstellen, Vorschau und Ausstellen eines Google Wallet Generic Pass (GenericClass + GenericObject) inkl. Save-Link (JWT), ohne externe NPM-Abhängigkeiten.
- Primärer Fluss: Benutzer konfiguriert Layout und Inhalte → Vorschau erzeugen → Save-Link generieren (JWT signiert mit Service-Account) → in Google Wallet speichern.

## Ubiquitous Language (Begriffe)
- IssuerId: Google Wallet Issuer Kennung.
- ClassId: Kennung der Generic Class (z. B. „employee_card_v2”).
- GenericClass (PassClass): Klassen-Definition mit Card-Layout (cardRowTemplateInfos).
- GenericObject (PassObject): Konkreter Pass mit Inhalten (Titel, Textmodule, Bilder, Barcode, Farbe, Links).
- TextModule: Ein Feld mit `id`, `header`, `body` (z. B. „telefon”).
- CardRow: Layout-Zeile. Typen: „oneItem” oder „twoItems” (links/rechts), referenziert TextModule per `fieldPath`.
- Save-Link: URL `https://pay.google.com/gp/v/save/<JWT>`.
- Service Account Key: JSON mit `private_key` + `client_email`.
- JWT (RS256): Signiertes Token mit `genericClasses` und `genericObjects`.

## Bounded Contexts
- Pass Authoring (UI): Erfasst Inhalte, pflegt TextModule-IDs, baut CardRows, triggert Vorschau/Save.
- Pass Issuing (Signatur): Baut Domain-Objekte (Class/Object), signiert JWT, erzeugt Save-Link.
- Configuration: Lädt `config.json`, stellt Default-Werte (Farben, Bilder, classRows) bereit, lädt Service-Account-Key.
- Support/Health: Health- und Konfig-Endpunkte für Betrieb/Debug.

## Aggregate, Entities und Value Objects
- Aggregate PassObject
  - Entities: TextModule[]
  - Value Objects: Barcode(type, value, altText), Color(hexBackgroundColor), Image(uri, description), Link(uri, description)
  - Invarianten: gültige `classId`, eindeutige/valide TextModule-IDs, gültiges Farbformat, optionale Bild-URIs.

- Aggregate PassClass
  - Entities: CardRow[] (oneItem/twoItems)
  - Invarianten: `issuerId.classId` gesetzt; CardRows referenzieren existierende TextModule-IDs.

## Domain Services
- ClassFactory: Erzeugt GenericClass aus Defaults + Overrides.
- ObjectFactory: Erzeugt GenericObject inkl. Normalisierung von TextModules und Fallbacks.
- IdGenerator: Generiert Objekt-IDs (Zeitstempel + random Suffix) falls nicht angegeben.
- JwtService: Baut RS256-JWT („savetoandroidpay“) und liefert Save-Link.

## Application Services / Use Cases
- Preview Class: Liefert Klassen-JSON (Layout) für UI/Debug.
- Preview Object: Liefert Objekt-JSON (Inhalte) für UI/Debug.
- Generate Save Link: Baut Class+Object, signiert JWT, liefert `saveUrl` + `objectPreview`.
- Get Config: Liefert sichere Sicht auf aktive Konfiguration.
- Health Check: Einfacher Livetest.

## API Endpoints (Adapter → Application)
- GET `/health`: System-Health.
- GET `/config`: Aktive Konfiguration (sicherer Ausschnitt).
- POST `/class/preview`: Eingabe: optional `issuerId`, `classId`, `classRows`; Ausgabe: GenericClass.
- POST `/object/preview`: Eingabe: Inhalte + TextModules + Layout; Ausgabe: GenericObject.
- POST `/wallet/save`: Eingabe: wie Preview + optional Overrides; Ausgabe: `{ saveUrl, objectPreview }`.

## Invarianten und Validierung (Kernregeln)
- issuerId/classId müssen gesetzt sein, bevor Save-Link erzeugt wird.
- Service-Account-Key muss geladen werden (private_key, client_email vorhanden) für `wallet/save`.
- TextModule-IDs sollten eindeutig und slug‑fähig sein; falls leer, werden sie aus Headern abgeleitet (slugify).
- CardRows dürfen nur auf vorhandene TextModule-IDs referenzieren.
- `hexBackgroundColor` ist ein valides Hex (z. B. `#4285f4`).
- Bild-URIs (Logo/Hero) sollten öffentlich erreichbar (https) sein.

## Kontext-Interaktionen (Context Map)
- Authoring → Issuing: Übergibt normalisierte Inhalte (TextModules, Layout) an Use Cases.
- Issuing → Google Wallet: Externer Boundary/ACL über JWT Save-Link (keine direkte API‑Schreiboperation ohne Tokenfluss).
- Configuration ist „Shared Kernel“ für Defaults und Secrets‑Pfad.

## Fehler- und Randfälle
- Fehlende Konfiguration: Fallback auf Defaults; Save-Link verweigert bis IDs/Key vorhanden.
- Ungültiger Key: 500 mit Hinweis zur `serviceAccountKeyPath`.
- Fehlende IDs: 400 bei `/wallet/save` wenn `issuerId`/`classId` fehlen.

## Erweiterungen (später)
- Validierungsschicht stärken (IDs, URLs, Farbschema) mit präzisen Fehlermeldungen.
- Persistenz (z. B. Speicherung erstellter Objekte, Audit‑Trail).
- Mehrsprachigkeit der Texte und ContentDescriptions.
- Mehrere Bounded Contexts für verschiedene Pass‑Typen (Tickets, Loyalty etc.) als eigene Klassen.
- Outbox/Domain Events (z. B. `SaveLinkGenerated`) für Integrationen.

## Technische Zuordnung (Mapping Code → DDD, informell)
- Endpoints/Adapter: `server.js` (HTTP‑Server, Routen, JSON‑Antworten)
- Factories/Services (Domänenlogik):
  - `buildClassJson(...)` – ClassFactory
  - `buildObjectJson(...)` – ObjectFactory + IdGenerator + Normalisierung
  - `createJwt(...)` – JwtService (RS256)
- UI/Authoring: `index.html` (Formular, Layout‑Builder, API‑Aufrufe)
- Konfiguration/Secrets: `config.json`, `config.example.json`, `key.json`

Diese DDD‑Skizze bildet den aktuellen Repo‑Stand ab und dient als Basis für weitere Modellierung (z. B. formale Aggregate‑Grenzen, explizite Domain‑Events, Kontext‑Diagramm).

