<?php
session_start();

// 1. Sicherheitscheck: Ist der Benutzer eingeloggt?
if (!isset($_SESSION['is_admin_logged_in']) || $_SESSION['is_admin_logged_in'] !== true) {
    $_SESSION['delete_message'] = "Zugriff verweigert. Bitte einloggen.";
    // Versuche, zum vorherigen Ordner zurückzuleiten, falls möglich
    $fallbackFolder = $_GET['folder'] ?? 'layers'; // Fallback, falls folder nicht gesetzt
    header("Location: view_folder.php?folder=" . urlencode($fallbackFolder));
    exit;
}

// 2. Erlaubte Basisordner (aus denen gelöscht werden darf)
$allowedBaseFolders = [
    'layers',
    'generated_images'
];

// 3. Parameter abrufen und validieren
$folderKey = $_GET['folder'] ?? ''; // Der Basisordner (z.B. 'layers')
$itemName  = $_GET['file'] ?? $_GET['item'] ?? ''; // Der zu löschende Dateiname oder Ordnername
$itemType  = $_GET['type'] ?? ''; // 'file' oder 'dir'

$redirectUrl = "view_folder.php?folder=" . urlencode($folderKey); // Standard-Weiterleitungs-URL

if (empty($folderKey) || empty($itemName) || empty($itemType)) {
    $_SESSION['delete_message'] = "Fehlende Parameter für Löschvorgang.";
    header("Location: " . ($folderKey ? $redirectUrl : "index.php")); // Wenn folderKey fehlt, zur Hauptseite
    exit;
}

// 4. Sicherheitscheck: Ist der angeforderte Basisordner erlaubt?
if (!in_array($folderKey, $allowedBaseFolders)) {
    $_SESSION['delete_message'] = "Löschen aus diesem Ordner ('" . htmlspecialchars($folderKey) . "') ist nicht erlaubt.";
    header("Location: index.php"); // Sicherer Fallback zur Hauptseite
    exit;
}

// 5. Sicherheitscheck: Dateinamen/Ordnernamen bereinigen (Path Traversal)
$itemName = basename($itemName);
if (empty($itemName) || $itemName === '.' || $itemName === '..') {
    $_SESSION['delete_message'] = "Ungültiger Datei- oder Ordnername zum Löschen.";
    header("Location: " . $redirectUrl);
    exit;
}

// 6. Vollständigen Pfad zum Element konstruieren
$basePath = dirname(__DIR__) . '/' . $folderKey . '/';
$fullPath = $basePath . $itemName;


// 7. Prüfen, ob das Element existiert, bevor versucht wird zu löschen
if (!file_exists($fullPath)) {
    $_SESSION['delete_message'] = "Das Element '" . htmlspecialchars($itemName) . "' wurde nicht gefunden und konnte nicht gelöscht werden.";
    header("Location: " . $redirectUrl);
    exit;
}

// 8. Eigentlicher Löschvorgang
try {
    if ($itemType === 'file') {
        if (is_file($fullPath)) {
            if (unlink($fullPath)) {
                $_SESSION['delete_message'] = "Datei '" . htmlspecialchars($itemName) . "' erfolgreich gelöscht.";
            } else {
                $_SESSION['delete_message'] = "Fehler beim Löschen der Datei '" . htmlspecialchars($itemName) . "'. Überprüfen Sie die Berechtigungen.";
            }
        } else {
            $_SESSION['delete_message'] = "'" . htmlspecialchars($itemName) . "' ist keine Datei.";
        }
    } elseif ($itemType === 'dir') {
        if (is_dir($fullPath)) {
            // Funktion zum rekursiven Löschen von Ordnern
            if (deleteDirectoryRecursive($fullPath)) {
                $_SESSION['delete_message'] = "Ordner '" . htmlspecialchars($itemName) . "' und sein Inhalt erfolgreich gelöscht.";
            } else {
                $_SESSION['delete_message'] = "Fehler beim Löschen des Ordners '" . htmlspecialchars($itemName) . "'. Überprüfen Sie die Berechtigungen oder ob der Ordner leer ist.";
            }
        } else {
            $_SESSION['delete_message'] = "'" . htmlspecialchars($itemName) . "' ist kein Ordner.";
        }
    } else {
        $_SESSION['delete_message'] = "Ungültiger Elementtyp zum Löschen.";
    }
} catch (Exception $e) {
    $_SESSION['delete_message'] = "Ein unerwarteter Fehler ist aufgetreten: " . $e->getMessage();
}

header("Location: " . $redirectUrl);
exit;


/**
 * Löscht einen Ordner und seinen gesamten Inhalt rekursiv.
 * VORSICHT: Diese Funktion löscht ohne weitere Nachfrage!
 * @param string $dir Der Pfad zum zu löschenden Verzeichnis.
 * @return bool True bei Erfolg, False bei Fehler.
 */
function deleteDirectoryRecursive(string $dir): bool {
    if (!is_dir($dir)) {
        return false;
    }
    // Sicherstellen, dass der Pfad mit einem Slash endet, um Probleme mit Verkettung zu vermeiden
    if (substr($dir, -1) !== '/') {
        $dir .= '/';
    }

    $items = glob($dir . '*', GLOB_MARK); // GLOB_MARK fügt einen Slash an Verzeichnisse an
    foreach ($items as $item) {
        if (is_dir($item)) {
            if (!deleteDirectoryRecursive($item)) { // Rekursiver Aufruf für Unterverzeichnisse
                return false; // Fehler beim Löschen des Unterverzeichnisses
            }
        } else {
            if (!unlink($item)) { // Datei löschen
                return false; // Fehler beim Löschen der Datei
            }
        }
    }
    // Nachdem alle Inhalte gelöscht wurden, das Verzeichnis selbst löschen
    return rmdir(rtrim($dir, '/'));
}

?>