<?php
session_start();

// 1. Sicherheitscheck: Ist der Benutzer eingeloggt?
if (!isset($_SESSION['is_admin_logged_in']) || $_SESSION['is_admin_logged_in'] !== true) {
    $_SESSION['user_message'] = "Zugriff verweigert. Bitte einloggen.";
    header("Location: index.php"); // Zurück zur Haupt-Loginseite
    exit;
}

// 2. Erlaubte Zielordner für Uploads und ihre Konfigurationen
$allowedUploadTargets = [
    'layers' => [
        'path' => dirname(__DIR__) . '/layers/', // Serverpfad
        'allowed_extensions' => ['jpg', 'jpeg', 'png', 'gif', 'webp'],
        'max_file_size_mb' => 5 // Maximale Dateigröße in MB
    ]
    // Hier könnten weitere Ordner für Uploads definiert werden, z.B. 'generated_images'
    // 'generated_images' => [
    //     'path' => dirname(__DIR__) . '/generated_images/',
    //     'allowed_extensions' => ['jpg', 'png'],
    //     'max_file_size_mb' => 2
    // ]
];

$targetFolderKey = $_POST['target_folder'] ?? '';
$redirectUrl = "view_folder.php?folder=" . urlencode($targetFolderKey); // Standard-Weiterleitung

// 3. Überprüfen, ob das Formular gesendet wurde und die Datei vorhanden ist
if (isset($_POST['submit_upload']) && isset($_FILES['uploaded_file'])) {

    $file = $_FILES['uploaded_file'];

    // 4. Zielordner-Validierung
    if (empty($targetFolderKey) || !array_key_exists($targetFolderKey, $allowedUploadTargets)) {
        $_SESSION['user_message'] = "Ungültiger oder nicht erlaubter Zielordner für den Upload.";
        header("Location: index.php"); // Sicherer Fallback
        exit;
    }

    $targetConfig = $allowedUploadTargets[$targetFolderKey];
    $uploadPath = $targetConfig['path'];

    // Sicherstellen, dass der Zielordner existiert und beschreibbar ist
    if (!is_dir($uploadPath) || !is_writable($uploadPath)) {
        $_SESSION['user_message'] = "Upload-Zielordner existiert nicht oder ist nicht beschreibbar.";
        header("Location: " . $redirectUrl);
        exit;
    }

    // 5. Fehlerprüfung beim Upload (PHP-interne Fehler)
    if ($file['error'] !== UPLOAD_ERR_OK) {
        $uploadErrors = [
            UPLOAD_ERR_INI_SIZE   => "Die hochgeladene Datei überschreitet die upload_max_filesize-Direktive in php.ini.",
            UPLOAD_ERR_FORM_SIZE  => "Die hochgeladene Datei überschreitet die MAX_FILE_SIZE-Direktive, die im HTML-Formular angegeben wurde.",
            UPLOAD_ERR_PARTIAL    => "Die hochgeladene Datei wurde nur teilweise hochgeladen.",
            UPLOAD_ERR_NO_FILE    => "Es wurde keine Datei hochgeladen.",
            UPLOAD_ERR_NO_TMP_DIR => "Fehlender temporärer Ordner.",
            UPLOAD_ERR_CANT_WRITE => "Fehler beim Schreiben der Datei auf die Festplatte.",
            UPLOAD_ERR_EXTENSION  => "Eine PHP-Erweiterung hat den Datei-Upload gestoppt.",
        ];
        $errorMessage = $uploadErrors[$file['error']] ?? "Unbekannter Upload-Fehler.";
        $_SESSION['user_message'] = "Upload-Fehler: " . $errorMessage;
        header("Location: " . $redirectUrl);
        exit;
    }

    // 6. Dateinamen bereinigen und Extension extrahieren
    $fileName = $file['name'];
    // Ersetze ungültige Zeichen und Leerzeichen, mache alles Kleinschreibung
    $safeFileName = preg_replace('/[^a-zA-Z0-9_.\-]/', '_', strtolower(basename($fileName)));
    $safeFileName = preg_replace('/\s+/', '_', $safeFileName); // Leerzeichen durch Unterstrich
    if (empty($safeFileName) || $safeFileName === '.' || $safeFileName === '..') {
        $_SESSION['user_message'] = "Ungültiger Dateiname nach Bereinigung.";
        header("Location: " . $redirectUrl);
        exit;
    }

    $fileExtension = strtolower(pathinfo($safeFileName, PATHINFO_EXTENSION));

    // 7. Dateityp-Validierung (anhand der Extension)
    if (!in_array($fileExtension, $targetConfig['allowed_extensions'])) {
        $_SESSION['user_message'] = "Ungültiger Dateityp. Erlaubt sind: " . implode(', ', $targetConfig['allowed_extensions']);
        header("Location: " . $redirectUrl);
        exit;
    }
    // Zusätzliche Prüfung mit getimagesize für echte Bilddateien (optional aber empfohlen)
    if (in_array($fileExtension, ['jpg', 'jpeg', 'png', 'gif', 'webp'])) {
        if (getimagesize($file['tmp_name']) === false) {
            $_SESSION['user_message'] = "Die Datei scheint kein gültiges Bild zu sein, obwohl die Endung passt.";
            header("Location: " . $redirectUrl);
            exit;
        }
    }


    // 8. Dateigrößen-Validierung
    $maxSizeBytes = $targetConfig['max_file_size_mb'] * 1024 * 1024;
    if ($file['size'] > $maxSizeBytes) {
        $_SESSION['user_message'] = "Datei ist zu groß. Maximale Größe: " . $targetConfig['max_file_size_mb'] . " MB.";
        header("Location: " . $redirectUrl);
        exit;
    }

    // 9. Zielpfad erstellen und auf Dateikonflikte prüfen
    $destinationPath = $uploadPath . $safeFileName;
    $counter = 1;
    $originalNameWithoutExt = pathinfo($safeFileName, PATHINFO_FILENAME);
    while (file_exists($destinationPath)) {
        // Wenn Datei existiert, neuen Namen generieren (z.B. bild_1.jpg, bild_2.jpg)
        $safeFileName = $originalNameWithoutExt . "_" . $counter . "." . $fileExtension;
        $destinationPath = $uploadPath . $safeFileName;
        $counter++;
        if ($counter > 100) { // Sicherheitslimit, um Endlosschleifen zu vermeiden
            $_SESSION['user_message'] = "Zu viele Dateien mit ähnlichem Namen. Bitte benennen Sie Ihre Datei um.";
            header("Location: " . $redirectUrl);
            exit;
        }
    }

    // 10. Datei verschieben
    if (move_uploaded_file($file['tmp_name'], $destinationPath)) {
        $_SESSION['user_message'] = "Datei '" . htmlspecialchars($safeFileName) . "' erfolgreich hochgeladen.";
    } else {
        $_SESSION['user_message'] = "Fehler beim Speichern der hochgeladenen Datei.";
    }

} else {
    $_SESSION['user_message'] = "Keine Datei zum Hochladen ausgewählt oder Formular nicht korrekt gesendet.";
}

header("Location: " . $redirectUrl);
exit;
?>