// js/uiController.js
import { layerListElement, baseImageSizeInfo } from './config.js';
// Benötigt Zugriff auf `layers` und `activeLayerIndex` zum Rendern.
// Und Callbacks, um Interaktionen (Klick auf Layer-Item) an main.js zu melden.

let selectLayerCallback = (index) => { console.warn("selectLayerCallback nicht initialisiert"); }; // Wird von main.js gesetzt
let deleteLayerCallback = (index) => { console.warn("deleteLayerCallback nicht initialisiert"); };   // Neuer Callback für Löschen
let toggleVisibilityCallback = (index) => { console.warn("toggleVisibilityCallback nicht initialisiert"); }; // Neuer Callback für Sichtbarkeit
let dragStartCallback = (startIndex) => { console.warn("dragStartCallback nicht initialisiert"); };
let dropCallback = (startIndex, endIndex) => { console.warn("dropCallback nicht initialisiert"); };

let draggedItemIndex = null; // Um den Index des gezogenen Elements zu speichern

export function updateLayerList(currentLayers, currentActiveLayerIndex) {
    layerListElement.innerHTML = "";
    currentLayers.forEach((layer, index) => {
        const div = document.createElement("div");
        div.className = "list-group-item layer-list-item" + (index === currentActiveLayerIndex ? " active" : "");
        // div selbst ist nicht mehr direkt klickbar für Auswahl, das machen wir spezifischer
        // oder wir lassen es und der Klick auf die Buttons wird durch e.stopPropagation() behandelt
div.setAttribute('draggable', true); // MACHT DAS ELEMENT DRAGGABLE
        div.dataset.index = index; // Speichere den Index für leichten Zugriff

        // --- DRAG & DROP EVENT LISTENER ---
        div.addEventListener('dragstart', (e) => {
            draggedItemIndex = index; // Index des gezogenen Elements merken
            e.dataTransfer.effectAllowed = 'move';
            e.dataTransfer.setData('text/plain', index); // Wichtig für Firefox, um Dragging zu ermöglichen
            // console.log('dragstart:', index);
            div.style.opacity = '0.5'; // Visuelles Feedback für das gezogene Element
            if (dragStartCallback) dragStartCallback(index); // Optionaler Callback
        });

        div.addEventListener('dragover', (e) => {
            e.preventDefault(); // Notwendig, um 'drop' zu erlauben
            e.dataTransfer.dropEffect = 'move';
            // Visuelles Feedback für die Drop-Zone (z.B. Rand hervorheben)
            // Hier könnte man prüfen, ob über der oberen oder unteren Hälfte gedroppt wird,
            // um eine Einfügemarkierung zu zeigen. Fürs Erste markieren wir das ganze Element.
            div.classList.add('drag-over-target');
            // console.log('dragover target index:', index);
            return false;
        });

        div.addEventListener('dragleave', (e) => {
            div.classList.remove('drag-over-target');
        });

        div.addEventListener('drop', (e) => {
            e.preventDefault();
            e.stopPropagation(); // Verhindert, dass übergeordnete Elemente das Drop-Event auch bekommen
            div.classList.remove('drag-over-target');

            const droppedOnItemIndex = index; // Index des Ziels
            // console.log('drop: dragged', draggedItemIndex, 'onto', droppedOnItemIndex);

            if (draggedItemIndex !== null && draggedItemIndex !== droppedOnItemIndex) {
                if (dropCallback) dropCallback(draggedItemIndex, droppedOnItemIndex);
            }
            draggedItemIndex = null; // Zurücksetzen
            return false;
        });

        div.addEventListener('dragend', (e) => {
            // console.log('dragend');
            div.style.opacity = '1'; // Visuelles Feedback zurücksetzen
            // Entferne ggf. 'drag-over-target' von allen Elementen (sicherheitshalber)
            layerListElement.querySelectorAll('.drag-over-target').forEach(el => el.classList.remove('drag-over-target'));
            draggedItemIndex = null; // Sicherstellen, dass es zurückgesetzt ist
        });
        // --- ENDE DRAG & DROP EVENT LISTENER ---
        
        const nameAndCoordsContainer = document.createElement('div');
        nameAndCoordsContainer.className = 'layer-item-name-coords';
        nameAndCoordsContainer.addEventListener('click', () => selectLayerCallback(index)); // Auswahl beim Klick auf Namen/Koordinaten


        const nameSpan = document.createElement("span");
        nameSpan.textContent = layer.label || `Layer ${index + 1}`;
        nameSpan.style.display = 'block'; // Damit Koordinaten darunter stehen
        nameAndCoordsContainer.appendChild(nameSpan);

        const bounds = layer.getCanvasBounds();
        const coordinatesSpan = document.createElement("span");
        coordinatesSpan.className = "layer-coordinates";
        coordinatesSpan.textContent =
            `X: ${bounds.x.toFixed(1)}, Y: ${bounds.y.toFixed(1)}, ` +
            `B: ${bounds.width.toFixed(1)}, H: ${bounds.height.toFixed(1)}`;
        nameAndCoordsContainer.appendChild(coordinatesSpan);
        div.appendChild(nameAndCoordsContainer);


        // Aktionen-Container
        const actionsDiv = document.createElement('div');
        actionsDiv.className = 'layer-item-actions';

        // --- NEUE BUTTONS FÜR REIHENFOLGE ---
        // Button "Nach oben"
        const upBtn = document.createElement('button');
        upBtn.className = 'btn-action move-layer-up-btn';
        upBtn.setAttribute('data-bs-toggle', 'tooltip');
        upBtn.setAttribute('data-bs-placement', 'top');
        upBtn.setAttribute('title', 'Layer eine Ebene nach oben');
        upBtn.innerHTML = '<span class="material-icons">arrow_upward</span>';
        if (index === 0) { // Erster Layer kann nicht weiter nach oben
            upBtn.disabled = true;
            upBtn.style.opacity = 0.5; // Visuelles Feedback für Deaktivierung
        }
        upBtn.addEventListener('click', (e) => {
            e.stopPropagation();
            moveLayerUpCallback(index);
        });
        actionsDiv.appendChild(upBtn); // Füge ihn vor Sichtbarkeit/Löschen ein oder wo es passt

        // Button "Nach unten"
        const downBtn = document.createElement('button');
        downBtn.className = 'btn-action move-layer-down-btn';
        downBtn.setAttribute('data-bs-toggle', 'tooltip');
        downBtn.setAttribute('data-bs-placement', 'top');
        downBtn.setAttribute('title', 'Layer eine Ebene nach unten');
        downBtn.innerHTML = '<span class="material-icons">arrow_downward</span>';
        if (index === currentLayers.length - 1) { // Letzter Layer kann nicht weiter nach unten
            downBtn.disabled = true;
            downBtn.style.opacity = 0.5;
        }
        downBtn.addEventListener('click', (e) => {
            e.stopPropagation();
            moveLayerDownCallback(index);
        });
        actionsDiv.appendChild(downBtn);

        // Sichtbarkeits-Button
        const visibilityBtn = document.createElement('button');
        visibilityBtn.className = 'btn-action toggle-visibility-btn';
        visibilityBtn.setAttribute('title', layer.isVisible ? 'Layer ausblenden' : 'Layer einblenden');
        visibilityBtn.innerHTML = `<span class="material-icons">${layer.isVisible ? 'visibility' : 'visibility_off'}</span>`;
        if (!layer.isVisible) {
            visibilityBtn.classList.add('layer-hidden'); // Für spezielles Styling, falls gewünscht
             div.style.opacity = 0.6; // Den gesamten Listeneintrag leicht ausgrauen
        }
        visibilityBtn.addEventListener('click', (e) => {
            e.stopPropagation(); // Verhindert, dass der Klick auch den Layer auswählt
            console.log("Toggle Visibility Clicked, Index:", index); // DEBUG
            toggleVisibilityCallback(index);
        });
        actionsDiv.appendChild(visibilityBtn);

        // Löschen-Button
        const deleteBtn = document.createElement('button');
        deleteBtn.className = 'btn-action delete-layer-btn';
        deleteBtn.setAttribute('data-bs-toggle', 'tooltip'); // Tooltip hinzugefügt
        deleteBtn.setAttribute('data-bs-placement', 'top');   // Tooltip hinzugefügt
        deleteBtn.setAttribute('title', 'Layer löschen');
        deleteBtn.innerHTML = '<span class="material-icons">delete_outline</span>';
        deleteBtn.addEventListener('click', (e) => {
            e.stopPropagation(); // Verhindert, dass der Klick auch den Layer auswählt
            // console.log("Delete Button Clicked in UI, Index:", index, "Name:", layer.label || `Layer ${index + 1}`); // DEBUG
            // Rufe den Callback auf, der in main.js das Modal öffnet
            // und übergib den Index und den Namen des Layers.
            deleteLayerCallback(index, layer.label || `Layer ${index + 1}`);
        });
        actionsDiv.appendChild(deleteBtn);

        div.appendChild(actionsDiv);
        layerListElement.appendChild(div);
    });
}


export function displayBaseImageSize(width, height) {
    if (width && height) {
        baseImageSizeInfo.textContent = `Originalgröße: ${width} x ${height} px`;
    } else {
        baseImageSizeInfo.textContent = "";
    }
}

export function displayBaseImageError() {
    baseImageSizeInfo.textContent = "Fehler beim Laden des Bildes.";
}

// js/uiController.js
// ... (Importe, bestehende Callback-Definitionen) ...

let moveLayerUpCallback = (index) => { console.warn("moveLayerUpCallback nicht initialisiert"); };
let moveLayerDownCallback = (index) => { console.warn("moveLayerDownCallback nicht initialisiert"); };

// initUIController muss die neuen Callbacks entgegennehmen
export function initUIController(
    onLayerSelect, 
    onDeleteLayer, 
    onToggleVisibility, 
    onMoveLayerUp,
     onMoveLayerDown,
    onDragStart, onDrop
    ) {
    selectLayerCallback = onLayerSelect;
    deleteLayerCallback = onDeleteLayer;
    toggleVisibilityCallback = onToggleVisibility;
    moveLayerUpCallback = onMoveLayerUp;     // NEU
    moveLayerDownCallback = onMoveLayerDown; // NEU
    dragStartCallback = onDragStart; // NEU
    dropCallback = onDrop;         // NEU
}
