<?php

declare(strict_types=1);

namespace WOWGmbH\Wownaechtebuch\Controller;


/**
 * This file is part of the "WOW Naechtebuch" Extension for TYPO3 CMS.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 * (c) 2024 WOW-GmbH <technik@wow.gmbh>, WOW-GmbH
 */
use TYPO3\CMS\Extbase\Http\ForwardResponse;
use TYPO3\CMS\Core\Utility\DebugUtility;
use \TYPO3\CMS\Core\Utility\GeneralUtility;
/**
 * LetterController
 */
class LetterController extends \TYPO3\CMS\Extbase\Mvc\Controller\ActionController
{

    /**
     * letterRepository
     *
     * @var \WOWGmbH\Wownaechtebuch\Domain\Repository\LetterRepository
     */
    protected $letterRepository;
    
    /**
     * treatmentRunRepository
     *
     * @var \WOWGmbH\Wownaechtebuch\Domain\Repository\TreatmentRunRepository
     */
    protected $treatmentRunRepository;

    /**
     * letterRecordRepository
     *
     * @var \WOWGmbH\Wownaechtebuch\Domain\Repository\LetterRecordRepository
     */
    protected $letterRecordRepository;

    /** 
     * persistenceManager
     *
     * @var \TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager
     */
    protected $persistenceManager;
    
    public function __construct()
    {
    }

    /**
     * @param \WOWGmbH\Wownaechtebuch\Domain\Repository\LetterRecordRepository $letterRecordRepository
     */
    public function injectLetterRecordRepository(\WOWGmbH\Wownaechtebuch\Domain\Repository\LetterRecordRepository $letterRecordRepository)
    {
        $this->letterRecordRepository = $letterRecordRepository;
    }


    /**
     * @param \TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager $persistenceManager
     */
    public function injectPersistenceManager(\TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager $persistenceManager)
    {
        $this->persistenceManager = $persistenceManager;
    }

    /**
     * @param \WOWGmbH\Wownaechtebuch\Domain\Repository\TreatmentRunRepository $treatmentRunRepository
     */
    public function injectTreatmentRunRepository(\WOWGmbH\Wownaechtebuch\Domain\Repository\TreatmentRunRepository $treatmentRunRepository)
    {
        $this->treatmentRunRepository = $treatmentRunRepository;
    }

    /**
     * @param \WOWGmbH\Wownaechtebuch\Domain\Repository\LetterRepository $letterRepository
     */
    public function injectLetterRepository(\WOWGmbH\Wownaechtebuch\Domain\Repository\LetterRepository $letterRepository)
    {
        $this->letterRepository = $letterRepository;
    }

    /**
     * action index
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function indexAction(): \Psr\Http\Message\ResponseInterface
    {
        return $this->htmlResponse();
    }

    /**
     * action list
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function listAction(): \Psr\Http\Message\ResponseInterface
    {
        $letters = $this->letterRepository->findAll();
        $this->view->assign('letters', $letters);
        return $this->htmlResponse();
    }

    /**
     * action listForAdmin
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function listForAdminAction(bool $archived = false): \Psr\Http\Message\ResponseInterface
    {
        if($archived) $letters = $this->letterRepository->findArchived();
        else $letters = $this->letterRepository->findAllUnarchived();

        $lettersBase = [];

        foreach ($letters->toArray() as $key => $letter)
        {
            $lettersBase[] = 
            [
                'latestLetterRecord' => $this->letterRecordRepository->findLatestByLetter($letter),
                'letter' => $letter
            ];
        }
        $this->view->assign('isDefaultView', !$archived);
        $this->view->assign('lettersBase', $lettersBase);
        return $this->htmlResponse();
    }

    /**
     * action listForAdminByChild
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function listForAdminByChildAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Child $child, bool $archived = false): \Psr\Http\Message\ResponseInterface
    {
        $latestTreatmentRun = $this->treatmentRunRepository->findLatestByChild($child);
        if($archived) $letters = $this->letterRepository->findArchivedByTreatmentRun($latestTreatmentRun);
        else $letters = $this->letterRepository->findUnarchivedByTreatmentRun($latestTreatmentRun);

        $lettersBase = [];

        foreach ($letters->toArray() as $key => $letter)
        {
            $lettersBase[] = 
            [
                'latestLetterRecord' => $this->letterRecordRepository->findLatestByLetter($letter),
                'letter' => $letter
            ];
        }

        $this->view->assign('isDefaultView', !$archived);
        $this->view->assign('child', $child);
        $this->view->assign('lettersBase', $lettersBase);
        return $this->htmlResponse();
    }

    /**
     * action show
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\Letter $letter
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function showAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Letter $letter): \Psr\Http\Message\ResponseInterface
    {
        $this->view->assign('letter', $letter);
        $this->view->assign('letterRecord', $this->letterRecordRepository->findLatestByLetter($letter));

        return $this->htmlResponse();
    }

    /**
     * action new
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function newAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Child $child): \Psr\Http\Message\ResponseInterface
    {
        $latestTreatmentRun = $this->treatmentRunRepository->findLatestByChild($child);
        $newLetter = GeneralUtility::makeInstance(\WOWGmbH\Wownaechtebuch\Domain\Model\Letter::class);
        $newLetter->setTreatmentRun($latestTreatmentRun);
        $newLetter->setState(0);
        $this->letterRepository->add($newLetter);
        $this->persistenceManager->persistAll();
        return (new ForwardResponse('new'))
            ->withControllerName('LetterRecord')
            ->withArguments(['letter' => $newLetter]);
    }

    /**
     * action edit
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function editAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Letter $letter): \Psr\Http\Message\ResponseInterface
    {
        return (new ForwardResponse('new'))
            ->withControllerName('LetterRecord')
            ->withArguments(['letter' => $letter]);
    }

    
    
    // !! Letter are created in new action. Is this bad pratice and should these be swapped???
    /**
     * action create 
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\Letter $newLetter
     */
    // public function createAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Letter $newLetter): \Psr\Http\Message\ResponseInterface
    // {
    //     $this->letterRepository->add($newLetter);
    //     return (new ForwardResponse('listForAdminByChildAction'))
    //         ->withControllerName('Letter')
    //         ->withArguments(['child' => $newLetter->getTreatmentRun()->getChild()]);
    // }

    /**
     * action delete
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\Letter $letter
     */
    public function deleteAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Letter $letter): \Psr\Http\Message\ResponseInterface
    {
        $child = $letter->getTreatmentRun()->getChild();
        $this->letterRepository->remove($letter);
        return $this->redirect('listForAdminByChild', null, null, ['child'=> $child]);
    }

    /**
     * action archive
     * 
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("letter")
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\Letter $letter
     */
    public function archiveAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Letter $letter) : \Psr\Http\Message\ResponseInterface
    {
        $letter->setState(\WOWGmbH\Wownaechtebuch\Domain\Model\Letter::STATUS_ARCHIVIERT);
 
        $this->letterRepository->update($letter);
        return $this->redirect('listForAdmin');
    }

    /**
     * action unarchive
     * 
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("letter")
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\Letter $letter
     */
    public function unarchiveAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Letter $letter) : \Psr\Http\Message\ResponseInterface
    {
        if($letter->isSent()) $letter->setState(\WOWGmbH\Wownaechtebuch\Domain\Model\Letter::STATUS_VERSENDET);
        else $letter->setState(\WOWGmbH\Wownaechtebuch\Domain\Model\Letter::STATUS_FREIGABE);
        

        $this->letterRepository->update($letter);
        return $this->redirect('listForAdmin', null, null, ['archived'=> true]);
    }

    /**
     * action listForParentUser
     * 
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser
     * 
     */
    public function listForParentUserAction(\WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser, bool $archived = false) {
        $lettersBase = [];

        foreach ($parentUser->getChilds() as $key => $child) {


            $latestTreatmentRun = $this->treatmentRunRepository->findLatestByChild($child);

            
            if($archived) $letters = $this->letterRepository->findArchivedAndSentByTreatmentRun($latestTreatmentRun);
            else $letters = $this->letterRepository->findUnarchivedAndSentByTreatmentRun($latestTreatmentRun);
    
            foreach ($letters->toArray() as $key => $letter)
            {
                $lettersBase[] = 
                [
                    'latestLetterRecord' => $this->letterRecordRepository->findLatestByLetter($letter),
                    'letter' => $letter
                ];
            }
        }

        $this->view->assign('isDefaultView', !$archived);
        $this->view->assign('parentUser', $parentUser);
        $this->view->assign('lettersBase', $lettersBase);
        return $this->htmlResponse();
    }
}
