// js/canvasController.js
import { canvas, ctx, HANDLE_SIZE, CENTER_HANDLE_RADIUS } from './config.js';
// Wir brauchen hier Zugriff auf die globalen Variablen `layers`, `activeLayerIndex`, `baseImage` etc.
// Diese müssen wir entweder übergeben oder aus einem "State"-Modul importieren.
// Für den Anfang übergeben wir sie oder machen sie im main.js bekannt.
// Alternativ: main.js ruft Funktionen in canvasController auf und übergibt den State.

let localLayersRef = [];
let localActiveLayerIndexRef = -1;
let localBaseImageRef = null;
let localActionRef = null;
let localOffsetRef = { x: 0, y: 0 };
let localInitialDistanceRef = 0;
let localInitialScaleRef = 1;

let updateUICallback = () => {}; // Callback für UI-Updates

export function initCanvasController(layers, activeLayerIndex, baseImage, action, offset, initialDistance, initialScale, uiUpdateCb) {
    localLayersRef = layers;
    // Wichtig: activeLayerIndex ist ein Wert, kein Objekt. Änderungen in main müssen hier reflektiert werden.
    // Besser wäre es, wenn main.js die Referenz auf activeLayerIndex direkt hält und canvasController
    // über Getter/Setter oder Callbacks darauf zugreift oder es als Parameter für Aktionen bekommt.
    // Vereinfacht für jetzt: wir nehmen an, main.js aktualisiert diese Werte bei Bedarf.
    // Oder wir exportieren die Setter aus main.js und importieren sie hier.

    // Um es einfacher zu machen, exportieren wir Setter, die main.js aufrufen kann
    // oder wir machen die Variablen in main.js zugänglich und importieren sie.
    // Für dieses Beispiel gehen wir davon aus, dass main.js die Logik steuert und
    // die Funktionen hier mit den nötigen Infos aufruft.

    updateUICallback = uiUpdateCb;
}


export function drawCanvas(currentLayers, currentActiveLayerIndex, currentBaseImage) {
    ctx.clearRect(0, 0, canvas.width, canvas.height);
    if (currentBaseImage) {
        ctx.drawImage(currentBaseImage, 0, 0, canvas.width, canvas.height);
    }

    currentLayers.forEach((layer, i) => {
        if (!layer.isVisible) return;
        const bounds = layer.getCanvasBounds();
        ctx.drawImage(layer.image, bounds.x, bounds.y, bounds.width, bounds.height);

        if (i === currentActiveLayerIndex) {
            const corners = layer.getCorners();
            ctx.fillStyle = "rgba(255, 0, 0, 0.7)";
            ctx.strokeStyle = "red";
            ctx.lineWidth = 1;

            for (const pt of Object.values(corners)) {
                ctx.fillRect(pt.x - HANDLE_SIZE / 2, pt.y - HANDLE_SIZE / 2, HANDLE_SIZE, HANDLE_SIZE);
                ctx.strokeRect(pt.x - HANDLE_SIZE / 2, pt.y - HANDLE_SIZE / 2, HANDLE_SIZE, HANDLE_SIZE);
            }
            const center = layer.getCenter();
            ctx.beginPath();
            ctx.arc(center.x, center.y, CENTER_HANDLE_RADIUS, 0, 2 * Math.PI);
            ctx.fillStyle = "rgba(0, 0, 255, 0.7)";
            ctx.fill();
            ctx.strokeStyle = "blue";
            ctx.stroke();
            ctx.strokeStyle = 'rgba(255,0,0,0.7)';
            ctx.lineWidth = 2;
            ctx.strokeRect(bounds.x, bounds.y, bounds.width, bounds.height);
        }
    });
}

export function getMousePos(evt) {
    const rect = canvas.getBoundingClientRect();
    const scaleX = canvas.width / rect.width;
    const scaleY = canvas.height / rect.height;
    return {
        x: (evt.clientX - rect.left) * scaleX,
        y: (evt.clientY - rect.top) * scaleY
    };
}

export function getTouchPos(touch) {
    const rect = canvas.getBoundingClientRect();
    const scaleX = canvas.width / rect.width;
    const scaleY = canvas.height / rect.height;
    return {
        x: (touch.clientX - rect.left) * scaleX,
        y: (touch.clientY - rect.top) * scaleY
    };
}

export function getTouchDistance(touches) {
    const dx = touches[0].clientX - touches[1].clientX;
    const dy = touches[0].clientY - touches[1].clientY;
    return Math.sqrt(dx * dx + dy * dy);
}

// Die Event-Listener selbst werden wir in main.js einrichten,
// da sie den globalen State (layers, activeLayerIndex etc.) modifizieren
// und dann Funktionen aus canvasController und uiController aufrufen.
// canvasController stellt die "reinen" Funktionen bereit.