// js/notifications.js

// Referenz zum Container einmalig holen (könnte auch aus config.js kommen)
const notificationContainer = document.getElementById('notificationContainer');

/**
 * Zeigt eine Benachrichtigung an.
 * @param {string} message Die anzuzeigende Nachricht.
 * @param {string} type Typ der Benachrichtigung ('success', 'error', 'info', 'warning'). Standard: 'info'.
 * @param {number} duration Anzeigedauer in Millisekunden. 0 für unendlich (manuelles Schließen nötig). Standard: 3000ms.
 */
export function showNotification(message, type = 'info', duration = 3000) {
    if (!notificationContainer) {
        console.error("Notification container not found!");
        alert(message); // Fallback auf alert, wenn Container fehlt
        return;
    }

    const notification = document.createElement('div');
    notification.className = `notification ${type.toLowerCase()}`; // z.B. notification error
    // notification.textContent = message; // Einfacher Text

    // Für mehr Flexibilität (z.B. HTML in message) und den Schließen-Button:
    const messageSpan = document.createElement('span');
    messageSpan.innerHTML = message; // Erlaubt einfaches HTML in der Nachricht
    notification.appendChild(messageSpan);

    // Schließen-Button hinzufügen
    const closeBtn = document.createElement('button');
    closeBtn.className = 'close-btn';
    closeBtn.innerHTML = '×'; // HTML-Entität für "x"
    closeBtn.setAttribute('aria-label', 'Schließen');
    closeBtn.onclick = () => removeNotification(notification);
    notification.appendChild(closeBtn);


    notificationContainer.appendChild(notification);

    // Animation zum Einblenden (kurze Verzögerung, damit CSS-Transition greift)
    setTimeout(() => {
        notification.classList.add('show');
    }, 10); // Eine sehr kleine Verzögerung reicht oft aus

    // Automatisch entfernen nach 'duration', außer duration ist 0
    if (duration > 0) {
        setTimeout(() => {
            removeNotification(notification);
        }, duration);
    }
}

function removeNotification(notificationElement) {
    if (notificationElement && notificationElement.parentElement === notificationContainer) {
        notificationElement.classList.remove('show'); // Startet Ausblend-Animation

        // Entferne das Element aus dem DOM, nachdem die Animation abgeschlossen ist
        // Die CSS-Transition-Dauer ist 0.4s (400ms)
        setTimeout(() => {
            if (notificationElement.parentElement === notificationContainer) {
                 notificationContainer.removeChild(notificationElement);
            }
        }, 400);
    }
}

// Globale Fehlerbehandlung (optional, aber nützlich)
// window.addEventListener('error', function (event) {
//     // Verhindert, dass der Fehler nur in der Konsole landet
//     // event.preventDefault(); // Vorsicht: Kann Debugging erschweren
//     showNotification(`Ein unerwarteter Fehler ist aufgetreten: ${event.message}`, 'error', 5000);
//     // Hier könnte man den Fehler auch an einen Server loggen
// });

// window.addEventListener('unhandledrejection', function (event) {
//     // Für nicht abgefangene Promise-Rejections
//     // event.preventDefault();
//     const reason = event.reason instanceof Error ? event.reason.message : String(event.reason);
//     showNotification(`Ein nicht behandelter Promise-Fehler: ${reason}`, 'error', 5000);
// });