<?php
// list_layers.php

header('Content-Type: application/json');
header('Cache-Control: no-cache, must-revalidate');

$clientAccessPathToLayers = 'layers/'; // So greift der Browser auf die Bilder zu (relativ zum Web-Root)
$serverReadPathToLayers = __DIR__ . '/../layers'; // So liest PHP die Dateien (eine Ebene hoch, dann in 'layers')

$config = [
    'allowedExtensions' => ['png', 'jpg', 'jpeg', 'gif', 'svg'],
    'namePatterns' => [
        '/^\d{4}-.+?-(?:F|P)-/i' => '',
        '/^(?:bild|layer|logo|icon|entwurf)_/i' => '',
        '/[-_]/' => ' ',
    ],
];

$layerOptions = [];

// Stelle sicher, dass der Pfad normalisiert ist (keine doppelten Slashes etc.) und absolut ist
$actualServerReadPath = realpath($serverReadPathToLayers);

if ($actualServerReadPath === false || !is_dir($actualServerReadPath)) {
    echo json_encode(['error' => 'Layer directory not found or not accessible at server path: ' . htmlspecialchars($serverReadPathToLayers)]);
    http_response_code(500);
    exit;
}

try {
    $iterator = new DirectoryIterator($actualServerReadPath);

    foreach ($iterator as $fileinfo) {
        if ($fileinfo->isFile()) {
            $filename = $fileinfo->getFilename();
            $extension = strtolower($fileinfo->getExtension());

            if (in_array($extension, $config['allowedExtensions'])) {
                $displayName = pathinfo($filename, PATHINFO_FILENAME);

                foreach ($config['namePatterns'] as $pattern => $replacement) {
                    $displayName = preg_replace($pattern, $replacement, $displayName);
                }
                $displayName = ucwords(strtolower(trim($displayName)));

                $layerOptions[] = [
                    'value' => rtrim($clientAccessPathToLayers, '/') . '/' . $filename, // Pfad für den Client
                    'text' => !empty($displayName) ? $displayName : $filename
                ];
            }
        }
    }

    usort($layerOptions, function ($a, $b) {
        return strcasecmp($a['text'], $b['text']);
    });

    echo json_encode($layerOptions);

} catch (Exception $e) {
    error_log("Fehler beim Lesen des Layer-Verzeichnisses: " . $e->getMessage());
    echo json_encode(['error' => 'Server error while reading layers.']);
    http_response_code(500);
    exit;
}
?>