<?php
header('Content-Type: application/json'); // Wir antworten mit JSON

// Konfiguration
$uploadDir = '../generated_images/'; // Relativer Pfad vom PHP-Skript zum Speicherordner
                                 // '../' geht eine Ebene hoch aus dem 'php/' Ordner
                                 // Stelle sicher, dass dieser Ordner existiert und beschreibbar ist!
$filenamePrefix = 'user_image_';

// Erstelle das Verzeichnis, falls es nicht existiert
if (!is_dir($uploadDir)) {
    if (!mkdir($uploadDir, 0775, true)) { // 0775 sind typische Berechtigungen, true für rekursiv
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Fehler: Upload-Verzeichnis konnte nicht erstellt werden.']);
        exit;
    }
}

// Prüfe, ob Schreibrechte für das Verzeichnis bestehen
if (!is_writable($uploadDir)) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Fehler: Upload-Verzeichnis ist nicht beschreibbar. Überprüfe die Serverberechtigungen.']);
    exit;
}


// Daten vom POST-Request holen
// Wir erwarten die Bilddaten als Base64-String unter dem Schlüssel 'imageData'
if (isset($_POST['imageData'])) {
    $imageData = $_POST['imageData'];

    // Entferne den Data-URL-Präfix (z.B. "data:image/png;base64,")
    // und dekodiere die Base64-Daten
    if (preg_match('/^data:image\/(\w+);base64,/', $imageData, $type)) {
        $imageData = substr($imageData, strpos($imageData, ',') + 1);
        $type = strtolower($type[1]); // png, jpeg etc.

        if (!in_array($type, ['png', 'jpeg', 'jpg'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Ungültiger Bildtyp. Nur PNG und JPEG werden unterstützt.']);
            exit;
        }

        $imageData = base64_decode($imageData);
        if ($imageData === false) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Fehler beim Dekodieren der Base64-Bilddaten.']);
            exit;
        }
    } else {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Ungültiges Bilddatenformat. Data-URL erwartet.']);
        exit;
    }

    // Dateinamen generieren: YYYY-MM-DD-HH-mm
    // Zeitzone setzen, um sicherzustellen, dass die korrekte Zeit verwendet wird
    date_default_timezone_set('Europe/Berlin'); // Oder deine gewünschte Zeitzone
    $timestamp = date('Y-m-d-H-i-s'); // Füge Sekunden für mehr Eindeutigkeit hinzu
    $filename = $filenamePrefix . $timestamp . '.' . ($type === 'jpeg' ? 'jpg' : $type);
    $filepath = $uploadDir . $filename;

    // Bild auf dem Server speichern
    if (file_put_contents($filepath, $imageData)) {
        // Erfolg
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'message' => 'Bild erfolgreich auf dem Server gespeichert.',
            'filename' => $filename, // Name der gespeicherten Datei
            'filepath' => $filepath  // Serverseitiger Pfad (eher für Debugging)
        ]);
    } else {
        // Fehler beim Speichern
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Fehler beim Speichern des Bildes auf dem Server.']);
    }

} else {
    // Keine Bilddaten empfangen
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Keine Bilddaten empfangen.']);
}
?>