"""
Reminder App for Windows notifications.

Reads reminder definitions from a JSON configuration file and shows toast
notifications at the scheduled times. Designed to run at startup so you get
regular prompts throughout the day.
"""

from __future__ import annotations

import argparse
import json
import logging
import sys
import time
from dataclasses import dataclass
from datetime import datetime, time as dt_time, timedelta
from pathlib import Path
from typing import Iterable, List, Optional, Sequence, Set

try:
    from winotify import Notification, audio
except ImportError:  # pragma: no cover - dependency hint for interactive use
    Notification = None  # type: ignore[assignment]
    audio = None  # type: ignore[assignment]


APP_ID = "WOW Reminder"
DEFAULT_CONFIG_PATH = Path("config.json")
DEFAULT_CHECK_INTERVAL_SECONDS = 30

DAY_ALIASES = {
    "mon": 0,
    "monday": 0,
    "montag": 0,
    "tue": 1,
    "tuesday": 1,
    "dienstag": 1,
    "wed": 2,
    "wednesday": 2,
    "mittwoch": 2,
    "thu": 3,
    "thursday": 3,
    "donnerstag": 3,
    "fri": 4,
    "friday": 4,
    "freitag": 4,
    "sat": 5,
    "saturday": 5,
    "samstag": 5,
    "sun": 6,
    "sunday": 6,
    "sonntag": 6,
}


def parse_time(value: str) -> dt_time:
    """Parse a HH:MM string into a time object."""
    try:
        hour_str, minute_str = value.split(":")
        hour = int(hour_str)
        minute = int(minute_str)
        if not (0 <= hour <= 23 and 0 <= minute <= 59):
            raise ValueError
        return dt_time(hour=hour, minute=minute)
    except (ValueError, AttributeError):
        raise ValueError(f"Ungültige Zeitangabe '{value}'. Verwende Format HH:MM.") from None


def parse_days(days: Optional[Sequence[str]]) -> Optional[Set[int]]:
    """Convert a list of day names (English or German) into weekday indices."""
    if days is None:
        return None
    result: Set[int] = set()
    for item in days:
        key = str(item).strip().lower()
        if key not in DAY_ALIASES:
            valid = ", ".join(
                ["monday", "tuesday", "wednesday", "thursday", "friday", "saturday", "sunday"]
            )
            raise ValueError(f"Unbekannter Wochentag '{item}'. Erlaubt: {valid}")
        result.add(DAY_ALIASES[key])
    return result


@dataclass(frozen=True)
class ReminderConfig:
    title: str
    message: str
    start_time: dt_time
    interval: timedelta
    occurrences_per_day: Optional[int]
    active_days: Optional[Set[int]]

    @classmethod
    def from_dict(cls, raw: dict) -> "ReminderConfig":
        if "message" not in raw:
            raise ValueError("Jeder Reminder benötigt einen 'message'-Text.")

        title = raw.get("title") or "Erinnerung"
        start_time_value = raw.get("start_time")
        if start_time_value is None:
            raise ValueError("Jeder Reminder benötigt ein 'start_time'-Feld im Format HH:MM.")
        start_time = parse_time(start_time_value)

        interval_minutes = raw.get("interval_minutes")
        if interval_minutes is None:
            raise ValueError("Jeder Reminder benötigt ein 'interval_minutes'-Feld.")
        try:
            interval_minutes = int(interval_minutes)
        except (TypeError, ValueError):
            raise ValueError("'interval_minutes' muss eine ganze Zahl sein.")
        if interval_minutes <= 0:
            raise ValueError("'interval_minutes' muss größer als 0 sein.")

        occurrences_value = raw.get("occurrences_per_day")
        occurrences: Optional[int]
        if occurrences_value is None:
            occurrences = None
        else:
            try:
                occurrences = int(occurrences_value)
            except (TypeError, ValueError):
                raise ValueError("'occurrences_per_day' muss eine ganze Zahl sein.")
            if occurrences <= 0:
                raise ValueError("'occurrences_per_day' muss größer als 0 sein.")

        active_days = parse_days(raw.get("days"))

        return cls(
            title=title,
            message=str(raw["message"]),
            start_time=start_time,
            interval=timedelta(minutes=interval_minutes),
            occurrences_per_day=occurrences,
            active_days=active_days,
        )

    def schedule_for_day(self, date_obj) -> List[datetime]:
        """Generate all scheduled times for the given date."""
        times: List[datetime] = []
        current = datetime.combine(date_obj, self.start_time)
        occurrences_count = 0
        while current.date() == date_obj:
            times.append(current)
            occurrences_count += 1
            if self.occurrences_per_day and occurrences_count >= self.occurrences_per_day:
                break
            current += self.interval
        return times

    def next_occurrence(self, after: datetime) -> datetime:
        """Return the next reminder time strictly after the provided moment."""
        date_cursor = after.date()
        search_time = after
        while True:
            if self.active_days and date_cursor.weekday() not in self.active_days:
                date_cursor += timedelta(days=1)
                search_time = datetime.combine(date_cursor, dt_time.min)
                continue

            for candidate in self.schedule_for_day(date_cursor):
                if candidate > search_time:
                    return candidate

            date_cursor += timedelta(days=1)
            search_time = datetime.combine(date_cursor, dt_time.min)


class ReminderState:
    """Tracks next run for a reminder configuration."""

    def __init__(self, config: ReminderConfig, reference: datetime):
        self.config = config
        self.next_run = self.config.next_occurrence(reference - timedelta(seconds=1))

    def due(self, now: datetime) -> bool:
        return now >= self.next_run

    def advance(self, reference: datetime) -> None:
        self.next_run = self.config.next_occurrence(reference)


class ToastNotifier:
    """Wrapper around winotify with graceful fallback if dependency is missing."""

    def __init__(self, app_id: str):
        if Notification is None or audio is None:
            raise RuntimeError(
                "Winotify ist nicht installiert. Bitte führe 'pip install -r requirements.txt' aus."
            )
        self.app_id = app_id

    def send(self, reminder: ReminderConfig) -> None:
        toast = Notification(app_id=self.app_id, title=reminder.title, msg=reminder.message)
        toast.set_audio(audio.Default, loop=False)
        toast.set_duration("short")
        toast.show()


def load_configuration(path: Path) -> List[ReminderConfig]:
    try:
        raw = json.loads(path.read_text(encoding="utf-8"))
    except FileNotFoundError:
        raise SystemExit(f"Konfigurationsdatei '{path}' wurde nicht gefunden.")
    except json.JSONDecodeError as exc:
        raise SystemExit(f"Konfiguration enthält einen JSON-Fehler: {exc}") from exc

    reminders_raw = raw.get("reminders")
    if not reminders_raw or not isinstance(reminders_raw, Iterable):
        raise SystemExit("Die Konfigurationsdatei benötigt eine Liste 'reminders'.")

    reminders: List[ReminderConfig] = []
    for index, item in enumerate(reminders_raw, start=1):
        if not isinstance(item, dict):
            raise SystemExit(f"Reminder #{index} ist kein Objekt.")
        try:
            reminders.append(ReminderConfig.from_dict(item))
        except ValueError as exc:
            raise SystemExit(f"Fehler in Reminder #{index}: {exc}") from exc
    return reminders


def build_states(reminders: Sequence[ReminderConfig], reference: datetime) -> List[ReminderState]:
    states = [ReminderState(reminder, reference) for reminder in reminders]
    for state in states:
        logging.info(
            "Nächste Erinnerung '%s' um %s",
            state.config.title,
            state.next_run.strftime("%d.%m.%Y %H:%M"),
        )
    return states


def main() -> None:
    parser = argparse.ArgumentParser(description="Windows Reminder App")
    parser.add_argument(
        "--config",
        type=Path,
        default=DEFAULT_CONFIG_PATH,
        help="Pfad zur Konfigurationsdatei (Standard: config.json).",
    )
    parser.add_argument(
        "--check-interval",
        type=int,
        default=DEFAULT_CHECK_INTERVAL_SECONDS,
        help="Wie oft (Sekunden) die Konfiguration geprüft wird (Standard: 30).",
    )
    parser.add_argument(
        "--app-id",
        default=APP_ID,
        help="App-ID für die Windows-Benachrichtigungen.",
    )
    parser.add_argument(
        "--verbose",
        action="store_true",
        help="Aktiviere ausführlichere Log-Ausgabe.",
    )
    args = parser.parse_args()

    logging.basicConfig(
        level=logging.DEBUG if args.verbose else logging.INFO,
        format="%(asctime)s - %(levelname)s - %(message)s",
    )

    config_path: Path = args.config.expanduser().resolve()
    logging.info("Verwende Konfiguration: %s", config_path)

    reminders = load_configuration(config_path)
    if not reminders:
        raise SystemExit("Keine Reminder in der Konfiguration gefunden.")

    notifier = ToastNotifier(app_id=args.app_id)
    last_mtime = config_path.stat().st_mtime
    states = build_states(reminders, datetime.now())

    check_interval = max(5, args.check_interval)
    logging.info("Reminder-App läuft. Zum Beenden STRG+C drücken.")

    while True:
        now = datetime.now()

        try:
            current_mtime = config_path.stat().st_mtime
        except FileNotFoundError:
            logging.error("Konfigurationsdatei %s wurde gelöscht. Stoppe Programm.", config_path)
            raise SystemExit(1)

        if current_mtime != last_mtime:
            logging.info("Konfiguration geändert. Lade neu...")
            reminders = load_configuration(config_path)
            states = build_states(reminders, now)
            last_mtime = current_mtime
            # After reloading skip to next loop iteration to avoid stale notifications
            time.sleep(1)
            continue

        due_states = [state for state in states if state.due(now)]
        for state in due_states:
            logging.info("Sende Erinnerung: %s", state.config.title)
            try:
                notifier.send(state.config)
            except Exception as exc:  # pragma: no cover - defensive for runtime issues
                logging.exception("Fehler beim Anzeigen der Benachrichtigung: %s", exc)
            state.advance(now)
            logging.debug(
                "Nächste Erinnerung '%s' geplant für %s",
                state.config.title,
                state.next_run.strftime("%d.%m.%Y %H:%M"),
            )

        if not states:
            sleep_seconds = check_interval
        else:
            next_due = min(state.next_run for state in states)
            delta = (next_due - datetime.now()).total_seconds()
            sleep_seconds = max(1, min(check_interval, int(delta)))

        time.sleep(sleep_seconds)


if __name__ == "__main__":
    try:
        main()
    except KeyboardInterrupt:
        print("\nReminder-App beendet.")
