<?php

declare(strict_types=1);

namespace WOWGmbH\Wownaechtebuch\Controller;


/**
 * This file is part of the "WOW Naechtebuch" Extension for TYPO3 CMS.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 * (c) 2024 WOW-GmbH <technik@wow.gmbh>, WOW-GmbH
 */
use TYPO3\CMS\Core\Utility\DebugUtility;
use \TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Http\ForwardResponse;

/**
 * ChildController
 */
class ChildController extends \TYPO3\CMS\Extbase\Mvc\Controller\ActionController
{

    /**
     * childRepository
     *
     * @var \WOWGmbH\Wownaechtebuch\Domain\Repository\ChildRepository
     */
    protected $childRepository;
    public function __construct()
    {
    }
    
    /**
     * surveyRepository
     *
     * @var \WOWGmbH\Wownaechtebuch\Domain\Repository\SurveyRepository
     */
    protected $surveyRepository;

    /**
     * treatmentRunRepository
     *
     * @var \WOWGmbH\Wownaechtebuch\Domain\Repository\TreatmentRunRepository
     */
    protected $treatmentRunRepository;

    /**
     * surveyRecordRepository
     *
     * @var \WOWGmbH\Wownaechtebuch\Domain\Repository\SurveyRecordRepository
     */
    protected $surveyRecordRepository;

    /**
     * parentUserRepository
     *
     * @var \WOWGmbH\Wownaechtebuch\Domain\Repository\ParentUserRepository
     */
    protected $parentUserRepository;

    /**
     * productRecordRepository
     *
     * @var \WOWGmbH\Wownaechtebuch\Domain\Repository\ProductRecordRepository
     */
    protected $productRecordRepository;
    
        /** 
     * persistenceManager
     *
     * @var \TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager
     */
    protected $persistenceManager;

        /**
     * @param \TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager $persistenceManager
     */
    public function injectPersistenceManager(\TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager $persistenceManager)
    {
        $this->persistenceManager = $persistenceManager;
    }

    /**
     * @param \WOWGmbH\Wownaechtebuch\Domain\Repository\ProductRecordRepository $productRecordRepository
     */
    public function injectProductRecordRepository(\WOWGmbH\Wownaechtebuch\Domain\Repository\ProductRecordRepository $productRecordRepository)
    {
        $this->productRecordRepository = $productRecordRepository;
    }

    /**
     * @param \WOWGmbH\Wownaechtebuch\Domain\Repository\ParentUserRepository $parentUserRepository
     */
    public function injectParentUserRepository(\WOWGmbH\Wownaechtebuch\Domain\Repository\ParentUserRepository $parentUserRepository)
    {
        $this->parentUserRepository = $parentUserRepository;
    }

    /**
     * @param \WOWGmbH\Wownaechtebuch\Domain\Repository\SurveyRepository $surveyRepository
     */
    public function injectSurveyRepository(\WOWGmbH\Wownaechtebuch\Domain\Repository\SurveyRepository $surveyRepository)
    {
        $this->surveyRepository = $surveyRepository;
    }

    /**
     * @param \WOWGmbH\Wownaechtebuch\Domain\Repository\SurveyRecordRepository $surveyRecordRepository
     */
    public function injectSurveyRecordRepository(\WOWGmbH\Wownaechtebuch\Domain\Repository\SurveyRecordRepository $surveyRecordRepository)
    {
        $this->surveyRecordRepository = $surveyRecordRepository;
    }

    /**
     * @param \WOWGmbH\Wownaechtebuch\Domain\Repository\ChildRepository $childRepository
     */
    public function injectChildRepository(\WOWGmbH\Wownaechtebuch\Domain\Repository\ChildRepository $childRepository)
    {
        $this->childRepository = $childRepository;
    }

    /**
     * @param \WOWGmbH\Wownaechtebuch\Domain\Repository\TreatmentRunRepository $treatmentRunRepository
     */
    public function injectTreatmentRunRepository(\WOWGmbH\Wownaechtebuch\Domain\Repository\TreatmentRunRepository $treatmentRunRepository)
    {
        $this->treatmentRunRepository = $treatmentRunRepository;
    }

    /**
     * action index
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function indexAction(): \Psr\Http\Message\ResponseInterface
    {
        $children = $this->childRepository->findAll();
        $this->view->assign('children', $children);
        return $this->htmlResponse();
    }

    /**
     * action list
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function listAction(): \Psr\Http\Message\ResponseInterface
    {
        $children = $this->childRepository->findAll();
        $this->view->assign('children', $children);
        return $this->htmlResponse();
    }

    /**
     * action listForAdmin
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function listForAdminAction(bool $archived = false): \Psr\Http\Message\ResponseInterface
    {
        if ($archived)  $children = $this->childRepository->findArchived();
        else $children = $this->childRepository->findAllUnarchived();

        $this->view->assign('pluginName', $this->request->getPluginName());
        $this->view->assign('isDefaultView', !$archived);
        $this->view->assign('children', $children);
        return $this->htmlResponse();
    }

    /**
     * action listForAdmin
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function listForCMAction(): \Psr\Http\Message\ResponseInterface
    {
        $children = $this->childRepository->findAll();

        $registrationSurvey = $this->surveyRepository->findByUid($this->settings['registration-survey']);
        
        $this->view->assign('pluginName', $this->request->getPluginName());
        $this->view->assign('registrationSurvey', $registrationSurvey);
        $this->view->assign('children', $children);
        return $this->htmlResponse();
    }

    /**
     * action listForNaechtebuch
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function listForNaechtebuchAction(): \Psr\Http\Message\ResponseInterface
    {
        if(!$this->request->getAttribute('frontend.user')) return $this->redirect('noUser');
        
        $frontendUserUid = $this->request->getAttribute('frontend.user')->user['uid'];
        $parentUser = $this->parentUserRepository->findOneBy(['user' => $frontendUserUid]);
        
        if(!$parentUser) return $this->redirect('noUser');

        $this->view->assign('parentUser', $parentUser);
        return $this->htmlResponse();
    }

    /**
     * action listForSurvey
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function listForSurveyAction(): \Psr\Http\Message\ResponseInterface
    {
        if(!$this->request->getAttribute('frontend.user')) return $this->redirect('noUser');
        
        $frontendUserUid = $this->request->getAttribute('frontend.user')->user['uid'];
        $parentUser = $this->parentUserRepository->findOneBy(['user' => $frontendUserUid]);
        
        if(!$parentUser) return $this->redirect('noUser');
        $registrationSurvey = $this->surveyRepository->findByUid($this->settings['registration-survey']);

        $this->view->assign('pluginName', $this->request->getPluginName());
        $this->view->assign('registrationSurvey', $registrationSurvey);
        $this->view->assign('parentUser', $parentUser);
        return $this->htmlResponse();
    }

    /**
     * action show
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\Child $child
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function showAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Child $child): \Psr\Http\Message\ResponseInterface
    {
        $latestTreatmentRun = $this->treatmentRunRepository->findLatestByChild($child);
        $protocols = $this->surveyRecordRepository->findBy(['survey_name' => 'Protokoll']);
        $this->view->assign('pluginName', $this->request->getPluginName());
        $this->view->assign('surveyRecords', $protocols);
        $this->view->assign('child', $child);
        return $this->htmlResponse();
    }

    /**
     * action new
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function newAction(\WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser ): \Psr\Http\Message\ResponseInterface
    {
        $this->view->assign('parentUser', $parentUser);
        return $this->htmlResponse();
    }

    /**
     * action create
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\Child $newChild
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("child")
     * 
     */
    public function createAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Child $newChild): \Psr\Http\Message\ResponseInterface
    {       
        $treatmentRun = GeneralUtility::makeInstance(\WOWGmbH\Wownaechtebuch\Domain\Model\TreatmentRun::class);
        $treatmentRun->setNumber(0);
        $treatmentRun->setState(0);
        $treatmentRun->setChild($newChild);

        
        $newChild->addTreatmentRun($treatmentRun);
        $this->childRepository->add($newChild);

        $this->persistenceManager->persistAll();
        $docSurvey = $this->surveyRepository->findByUid($this->settings['doc-survey']);

        return (new ForwardResponse('fillRegistrationSurveyForCM'))
            ->withControllerName('SurveyRecord')
            ->withArguments(['survey'=> $docSurvey, 'child' => $newChild]);
        // return $this->redirect('editForCM', 'ParentUser', null, ['parentUser' => $newChild->getParentUser()]);
    }

    /**
     * action newForUser
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\ProductRecord $productRecord
     * @return \Psr\Http\Message\ResponseInterface
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("parentUser")
     */
    public function newForUserAction(\WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser, \WOWGmbH\Wownaechtebuch\Domain\Model\ProductRecord $productRecord = null ): \Psr\Http\Message\ResponseInterface
    {
        $this->view->assign('parentUser', $parentUser);
        $this->view->assign('productRecord', $productRecord);
        return $this->htmlResponse();
    }

    /**
     * action createForUser
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\Child $newChild
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\ProductRecord $productRecord
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("newChild")
     * 
     */
    public function createForUserAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Child $newChild, \WOWGmbH\Wownaechtebuch\Domain\Model\ProductRecord $productRecord = null): \Psr\Http\Message\ResponseInterface
    {       
        $productRecord->setChild($newChild);
        $treatmentRun = GeneralUtility::makeInstance(\WOWGmbH\Wownaechtebuch\Domain\Model\TreatmentRun::class);
        $treatmentRun->setNumber(0);
        $treatmentRun->setState(0);
        $treatmentRun->setChild($newChild);

        
        $newChild->addTreatmentRun($treatmentRun);
        $this->childRepository->add($newChild);
        $this->productRecordRepository->update($productRecord);

        $registrationSurvey = $this->surveyRepository->findByUid($this->settings['registration-survey']);

        DebugUtility::debug($registrationSurvey);
        $this->persistenceManager->persistAll();
        return (new ForwardResponse('fillRegistrationSurvey'))
            ->withControllerName('SurveyRecord')
            ->withArguments(['survey'=> $registrationSurvey, 'child' => $newChild]);
        // return (new ForwardResponse('editForUser'))
        //     ->withControllerName('ParentUser')
        //     ->withArguments(['parentUser' => $newChild->getParentUser()]);
    }

    
    /**
     * action listForTreatmentRun
     * 
     * 
     */
    public function listForTreatmentRunAction(): \Psr\Http\Message\ResponseInterface
    {
        if(!$this->request->getAttribute('frontend.user')) return $this->redirect('noUser');
        
        $frontendUserUid = $this->request->getAttribute('frontend.user')->user['uid'];
        $parentUser = $this->parentUserRepository->findOneBy(['user' => $frontendUserUid]);
        
        if(!$parentUser) return $this->redirect('noUser');
        $relapseSurvey = $this->surveyRepository->findByUid($this->settings['relapse-survey']);

        $this->view->assign('pluginName', $this->request->getPluginName());
        $this->view->assign('relapseSurvey', $relapseSurvey);
        $this->view->assign('parentUser', $parentUser);
        return $this->htmlResponse();
    }


    /**
     * action edit
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\Child $child
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("child")
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function editAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Child $child): \Psr\Http\Message\ResponseInterface
    {
        $this->view->assign('child', $child);
        return $this->htmlResponse();
    }

    /**
     * action editForCM
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\Child $child
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("child")
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function editForCMAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Child $child): \Psr\Http\Message\ResponseInterface
    {
        $this->view->assign('child', $child);
        return $this->htmlResponse();
    }

        /**
     * action editDocForCM
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\Child $newChild
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("child")
     * 
     */
    public function editDocForCMAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Child $child): \Psr\Http\Message\ResponseInterface
    {       
        // $treatmentRun = GeneralUtility::makeInstance(\WOWGmbH\Wownaechtebuch\Domain\Model\TreatmentRun::class);
        // $treatmentRun->setNumber(0);
        // $treatmentRun->setState(0);
        // $treatmentRun->setChild($newChild);

        
        // $newChild->addTreatmentRun($treatmentRun);
        // $this->childRepository->add($newChild);

        $this->persistenceManager->persistAll();
        $docSurvey = $this->surveyRepository->findByUid($this->settings['doc-survey']);

        return (new ForwardResponse('fillRegistrationSurveyForCM'))
            ->withControllerName('SurveyRecord')
            ->withArguments(['survey'=> $docSurvey, 'child' => $child]);
        // return $this->redirect('editForCM', 'ParentUser', null, ['parentUser' => $newChild->getParentUser()]);
    }

    /**
     * action update
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\Child $child
     */
    public function updateAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Child $child): \Psr\Http\Message\ResponseInterface
    {
        $this->childRepository->update($child);
        return $this->redirect('list');
    }

    /**
     * action updateForCM
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\Child $child
     */
    public function updateForCMAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Child $child): \Psr\Http\Message\ResponseInterface
    {
        $this->childRepository->update($child);
        return $this->redirect('editForCM', 'ParentUser', null, ['parentUser' => $child->getParentUser()]);
    }

    /**
     * action delete
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\Child $child
     */
    public function deleteAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Child $child): \Psr\Http\Message\ResponseInterface
    {
        $this->childRepository->remove($child);
        return $this->redirect('listForAdmin');
    }

    /**
     * action noUser
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function noUserAction(): \Psr\Http\Message\ResponseInterface
    {
        return $this->htmlResponse();
    }

    /**
     * action archive
     * 
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("child")
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\Child $child
     */
    public function archiveAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Child $child) : \Psr\Http\Message\ResponseInterface
    {
        $child->setState(\WOWGmbH\Wownaechtebuch\Domain\Model\Child::STATE_ARCHIVED);
 
        $this->childRepository->update($child);
        return $this->redirect('listForAdmin');
    }

    /**
     * action unarchive
     * 
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("child")
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\Child $child
     */
    public function unarchiveAction(\WOWGmbH\Wownaechtebuch\Domain\Model\Child $child) : \Psr\Http\Message\ResponseInterface
    {
        $child->setState(\WOWGmbH\Wownaechtebuch\Domain\Model\Child::STATE_ACTIVE);

        $this->childRepository->update($child);
        return $this->redirect('listForAdmin', null, null, ['archived'=> true]);
    }
}
