<?php

declare(strict_types=1);

namespace WOWGmbH\Wownaechtebuch\Controller;


/**
 * This file is part of the "WOW Naechtebuch" Extension for TYPO3 CMS.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 * (c) 2024 WOW-GmbH <technik@wow.gmbh>, WOW-GmbH
 */
use TYPO3\CMS\Core\Utility\DebugUtility;
use TYPO3\CMS\Extbase\Http\ForwardResponse;
use In2code\Femanager\Domain\Repository\UserGroupRepository;
use \WOWGmbH\Wownaechtebuch\Event\NewParentUserEvent;

/**
 * ParentUserController
 */
class ParentUserController extends \TYPO3\CMS\Extbase\Mvc\Controller\ActionController
{

    /**
     * productRecordRepository
     *
     * @var \WOWGmbH\Wownaechtebuch\Domain\Repository\ProductRecordRepository
     */
    protected $productRecordRepository;

    /**
     * parentUserRepository
     *
     * @var \WOWGmbH\Wownaechtebuch\Domain\Repository\ParentUserRepository
     */
    protected $parentUserRepository;

    /**
     * surveyRepository
     *
     * @var \WOWGmbH\Wownaechtebuch\Domain\Repository\SurveyRepository
     */
    protected $surveyRepository;
    
    /**
     * userGroupRepository
     *
     * @var \In2code\Femanager\Domain\Repository\UserGroupRepository
     */
    protected $userGroupRepository;

    public function __construct()
    {
    }

    /**
     * @param \WOWGmbH\Wownaechtebuch\Domain\Repository\SurveyRepository $surveyRepository
     */
    public function injectSurveyRepository(\WOWGmbH\Wownaechtebuch\Domain\Repository\SurveyRepository $surveyRepository)
    {
        $this->surveyRepository = $surveyRepository;
    }


    /**
     * @param \WOWGmbH\Wownaechtebuch\Domain\Repository\ParentUserRepository $parentUserRepository
     */
    public function injectParentUserRepository(\WOWGmbH\Wownaechtebuch\Domain\Repository\ParentUserRepository $parentUserRepository)
    {
        $this->parentUserRepository = $parentUserRepository;
    }

    /**
     * @param \WOWGmbH\Wownaechtebuch\Domain\Repository\UserGroupRepository $userGroupRepository
     */
    public function injectUserGroupRepository(\In2code\Femanager\Domain\Repository\UserGroupRepository $userGroupRepository)
    {
        $this->userGroupRepository = $userGroupRepository;
    }

    /**
     * @param \WOWGmbH\Wownaechtebuch\Domain\Repository\ProductRecordRepository $productRecordRepository
     */
    public function injectProductRecordRepository(\WOWGmbH\Wownaechtebuch\Domain\Repository\ProductRecordRepository $productRecordRepository)
    {
        $this->productRecordRepository = $productRecordRepository;
    }

    /**
     * action index
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function indexAction(): \Psr\Http\Message\ResponseInterface
    {
        $parentUsers = $this->parentUserRepository->findAll();
        $this->view->assign('parentUsers', $parentUsers);
        return $this->htmlResponse();
    }

    /**
     * action list
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function listAction(): \Psr\Http\Message\ResponseInterface
    {
        $parentUsers = $this->parentUserRepository->findAll();
        $this->view->assign('pluginName', $this->request->getPluginName());
        $this->view->assign('parentUsers', $parentUsers);
        return $this->htmlResponse();
    }

    /**
     * action listForAdmin
     *
     * @param bool $archived
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function listForAdminAction(bool $archived = false): \Psr\Http\Message\ResponseInterface
    {
        if ($archived)  $parentUsers = $this->parentUserRepository->findArchived();
        else $parentUsers = $this->parentUserRepository->findAllUnarchived();

        $this->view->assign('pluginName', $this->request->getPluginName());
        $this->view->assign('isDefaultView', !$archived);
        $this->view->assign('parentUsers', $parentUsers);
        return $this->htmlResponse();
    }

    /**
     * action listForAdmin
     *
     * @param bool $archived
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function listForCMAction(bool $archived = false): \Psr\Http\Message\ResponseInterface
    {
        if ($archived)  $parentUsers = $this->parentUserRepository->findArchived();
        else $parentUsers = $this->parentUserRepository->findAllUnarchived();

        $this->view->assign('pluginName', $this->request->getPluginName());
        $this->view->assign('isDefaultView', !$archived);
        $this->view->assign('parentUsers', $parentUsers);
        return $this->htmlResponse();
    }

    /**
     * action show
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function showAction(\WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser): \Psr\Http\Message\ResponseInterface
    {

        $this->view->assign('parentUser', $parentUser);
        $this->view->assign('pluginName', $this->request->getPluginName());
        return $this->htmlResponse();
    }

    /**
     * action showLoggedUser
     *
     * 
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function showLoggedUserAction(): \Psr\Http\Message\ResponseInterface
    {
        $frontendUserUid = $this->request->getAttribute('frontend.user')->user['uid'];
        $parentUser = $this->parentUserRepository->findOneBy(['user' => $frontendUserUid]);
        return $this->redirect('show', null, null, ['parentUser' => $parentUser]);

        // // return (new ForwardResponse('list'))
        // //     ->withArguments(['pluginName' => $this->request->getPluginName()]);
        //     // ->withArguments(['parentUser' => $parentUser]);
        //     // ->withArguments(['parentUser' => $parentUser, 'pluginName' => $this->request->getPluginName()]);
        
        // // return $this->htmlResponse();
    }

    /**
     * action new
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function newAction(): \Psr\Http\Message\ResponseInterface
    {
        return $this->htmlResponse();
    }

    /**
     * action create
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $newParentUser
     */
    public function createAction(\WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $newParentUser): \Psr\Http\Message\ResponseInterface
    {
        $this->parentUserRepository->add($newParentUser);
        return $this->redirect('listForCM');
    }

    /**
     * action edit
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("parentUser")
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function editAction(\WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser): \Psr\Http\Message\ResponseInterface
    {
        $this->view->assign('pluginName', $this->request->getPluginName());
        $this->view->assign('parentUser', $parentUser);
        return $this->htmlResponse();
    }

    /**
     * action editForCM
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("parentUser")
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function editForCMAction(\WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser): \Psr\Http\Message\ResponseInterface
    {
        $productRecords = [];

        foreach ( $parentUser->getChilds()->toArray() as $child) {
            $productRecords = array_merge($productRecords, $this->productRecordRepository->findBy(['child' => $child->getUid()])->toArray());
        }
        $registrationSurvey = $this->surveyRepository->findByUid($this->settings['registration-survey']);

        $this->view->assign('pluginName', $this->request->getPluginName());
        $this->view->assign('registrationSurvey', $registrationSurvey);
        $this->view->assign('parentUser', $parentUser);
        $this->view->assign('productRecords', $productRecords);
        return $this->htmlResponse();
    }

    /**
     * action editForUser
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("parentUser")
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function editForUserAction(\WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser): \Psr\Http\Message\ResponseInterface
    {
        $this->view->assign('pluginName', $this->request->getPluginName());
        $this->view->assign('parentUser', $parentUser);
        return $this->htmlResponse();
    }

    /**
     * action editForParentUser
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("parentUser")
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function editForParentUserAction(\WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser): \Psr\Http\Message\ResponseInterface
    {
        $this->view->assign('pluginName', $this->request->getPluginName());
        $this->view->assign('parentUser', $parentUser);
        return $this->htmlResponse();
    }

    /**
     * action updateForUser
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser
     */
    public function updateForUserAction(\WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser): \Psr\Http\Message\ResponseInterface
    {
        $userGroupUid = $this->settings['user-group'] ?? 1;
        // $pageUid =  intval( $this->settings['redirect-pid']) ?? 1;

        // After this the ParentUser is complete (because of Validation) and we can set the correct user group
        $userGroup = $this->userGroupRepository->findByUid($userGroupUid);
        $parentUser->getUser()->removeUsergroup(array_pop(array_reverse($parentUser->getUser()->getUsergroup()->toArray())));
        $parentUser->getUser()->addUsergroup($userGroup);

        /** @var NewParentUserEvent $event */
        $event = $this->eventDispatcher->dispatch(
            new NewParentUserEvent($parentUser),
        );
        
        // $parentUser->getUser()->getUsergroup()->removeAll();

        $this->parentUserRepository->update($parentUser);

        $uriBuilder = $this->uriBuilder;
        $uri = $uriBuilder
        ->setTargetPageUid(1)
        ->build();
        return $this->redirectToUri($uri, 0, 201);
    }

    /**
     * action update
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser
     */
    public function updateAction(\WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser): \Psr\Http\Message\ResponseInterface
    {
        $this->parentUserRepository->update($parentUser);
        return $this->redirect('list');
    }

    /**
     * action updateForParentUser
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser
     */
    public function updateForParentUserAction(\WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser): \Psr\Http\Message\ResponseInterface
    {
        $this->parentUserRepository->update($parentUser);
        return $this->redirect('showLoggedUser');
    }

    /**
     * action updateForCM
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser
     */
    public function updateForCMAction(\WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser): \Psr\Http\Message\ResponseInterface
    {
        $this->parentUserRepository->update($parentUser);
        return $this->redirect('listForCM');
    }

    /**
     * action archive
     * 
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("parentUser")
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser
     */
    public function archiveAction(\WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser) : \Psr\Http\Message\ResponseInterface
    {
        $parentUser->setState(\WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser::STATE_ARCHIVED);
 
        $this->parentUserRepository->update($parentUser);
        return $this->redirect('listForAdmin');
    }

    /**
     * action unarchive
     * 
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("parentUser")
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser
     */
    public function unarchiveAction(\WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser) : \Psr\Http\Message\ResponseInterface
    {
        $parentUser->setState(\WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser::STATE_ACTIVE);

        $this->parentUserRepository->update($parentUser);
        return $this->redirect('listForAdmin', null, null, ['archived'=> true]);
    }

    /**
     * action delete
     *
     * @param \WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser
     */
    public function deleteAction(\WOWGmbH\Wownaechtebuch\Domain\Model\ParentUser $parentUser): \Psr\Http\Message\ResponseInterface
    {
        $this->parentUserRepository->remove($parentUser);
        return $this->redirect('list');
    }
}
